CREATE OR REPLACE FUNCTION disk.get_migration_lock(uid bigint)
    RETURNS void
    LANGUAGE sql
AS
$$
SELECT pg_advisory_xact_lock(1, uid::bit(32)::int)
$$;

CREATE OR REPLACE FUNCTION disk.is_locked_for_migration(uid_for_lock bigint, until timestamp with time zone)
    RETURNS bool
    LANGUAGE plpgsql
AS
$$
DECLARE
    locked        BOOLEAN;
BEGIN
    IF (coalesce(current_setting('app.is_locked_for_migration', true), '') != '')
    THEN
        RETURN current_setting('app.is_locked_for_migration', true) = 'true';
    END IF;
    PERFORM pg_advisory_xact_lock_shared(1, uid_for_lock::bit(32)::int);
    SELECT exists(SELECT FROM disk.migration_lock WHERE uid = uid_for_lock AND lock_until > until) INTO locked;
    set local app.is_locked_for_migration to locked;
    RETURN locked;
END
$$;
CREATE OR REPLACE FUNCTION disk.guard_by_migration_lock()
    RETURNS TRIGGER
    LANGUAGE plpgsql
AS
$$
DECLARE
    column_name   TEXT;
    requested_uid BIGINT;
    locked        BOOLEAN;
BEGIN
    IF (disk.guard_by_migration_lock_enabled() AND coalesce(current_setting('app.disable_migration_lock', true), 'false') != 'true')
    THEN
        column_name := tg_argv[0];
        IF (TG_OP = 'UPDATE' OR TG_OP = 'INSERT')
        THEN
            EXECUTE 'SELECT $1.' || column_name INTO requested_uid USING NEW;
        ELSIF (TG_OP = 'DELETE')
        THEN
            EXECUTE 'SELECT $1.' || column_name INTO requested_uid USING OLD;
        ELSE
            RAISE EXCEPTION 'Other action occurred: %, at %', TG_OP, now();
        END IF;
        locked := disk.is_locked_for_migration(requested_uid, now());
        IF (locked)
        THEN
            RAISE EXCEPTION 'Can not write to table because user is locked' USING errcode = 'user locked for migration';
        end if;
    END IF;
    IF (TG_OP = 'DELETE')
    THEN
        RETURN OLD;
    ELSE
        RETURN NEW;
    END IF;
END ;
$$;

CREATE TRIGGER albums_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.albums
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER album_items_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.album_items
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER changelog_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.changelog
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER deletion_log_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.deletion_log
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER disk_info_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.disk_info
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER folders_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.folders
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER version_links_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.version_links
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER files_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.files
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER version_data_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.version_data
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER additional_file_links_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.additional_file_links
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER filesystem_locks_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.filesystem_locks
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER link_data_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.link_data
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER misc_data_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.misc_data
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER operations_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.operations
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER source_ids_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.source_ids
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER user_index_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.user_index
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');

CREATE TRIGGER user_activity_info_guard_by_migration_lock
    AFTER INSERT OR UPDATE OR DELETE
    ON disk.user_activity_info
    FOR EACH ROW
EXECUTE PROCEDURE disk.guard_by_migration_lock('uid');
