CREATE OR REPLACE FUNCTION disk.is_cycle_or_limit_folder(uid bigint, fid uuid, level_limit int)
	RETURNS int
	LANGUAGE plpgsql
AS $$
DECLARE
	to_cycle boolean;
	to_limit boolean;
BEGIN
    WITH RECURSIVE r AS (
	    SELECT 2						AS level,
	           dir.fid					AS sfid,
	           dir.parent_fid			AS pfid,
	           dir.parent_fid = dir.fid AS is_cycle,
	           false					AS is_limit
	      FROM disk.folders				AS dir
	     WHERE dir.uid = $1
	       AND dir.fid = $2
	     UNION ALL
	    SELECT child.level + 1,
	    	   child.sfid,
	    	   parent.parent_fid,
	    	   parent.parent_fid = child.sfid,
	    	   child.level >= $3
	      FROM r child JOIN disk.folders parent ON child.pfid = parent.fid
	     WHERE parent.uid = $1
	       AND parent.parent_fid IS NOT NULL
	       AND parent.fid <> child.sfid
	       AND child.level <= $3
	)
	SELECT is_limit, is_cycle INTO to_limit, to_cycle FROM r ORDER BY level DESC LIMIT 1;
	IF to_cycle THEN
		RETURN 1;
	ELSIF to_limit THEN
		RETURN 2;
	ELSE
		RETURN 0;
	END IF;
END $$;

CREATE OR REPLACE FUNCTION disk.check_cycle_or_limit_folder()
	RETURNS TRIGGER
	LANGUAGE plpgsql
AS $check_cycle_or_limit_folder$
DECLARE
	err_type int;
BEGIN
	err_type := disk.is_cycle_or_limit_folder(NEW.uid, NEW.fid, 1024);
	IF (err_type = 1) THEN
		RAISE EXCEPTION 'Cycling folder levels: "%". uid: "%", parent_fid: "%"', NEW.name, NEW.uid, NEW.parent_fid USING ERRCODE = 'unique_violation';
	ELSIF (err_type = 2) THEN
		RAISE EXCEPTION 'Exceeding the limit of nesting levels: "%". uid: "%", parent_fid: "%"', NEW.name, NEW.uid, NEW.parent_fid USING ERRCODE = 'unique_violation';
	ELSE
    	RETURN NULL;
    END IF;
END;
$check_cycle_or_limit_folder$;

CREATE TRIGGER check_cycle_or_limit_folder
    AFTER INSERT OR UPDATE OF parent_fid
    ON disk.folders
    FOR EACH ROW
EXECUTE PROCEDURE disk.check_cycle_or_limit_folder();
