ALTER TABLE job ADD CONSTRAINT check_task_length CHECK (char_length(task) <= 500);

CREATE TABLE job_counters (
    task            TEXT        NOT NULL,
    status          job_status  NOT NULL,
    tasks_count     bigint      NOT NULL,

    PRIMARY KEY (task, status)
);

CREATE TABLE job_counters_queue (
    id              serial      NOT NULL,
    task            TEXT        NOT NULL,
    status          job_status  NOT NULL,
    count_change    bigint      NOT NULL
);

CREATE OR REPLACE FUNCTION trigger_after_insert_job () RETURNS trigger AS $$
BEGIN
    if (NEW.status != 'completed') then
        INSERT INTO job_counters_queue
            (task, status, count_change) VALUES
            (NEW.task, NEW.status, 1);
    end if;
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION trigger_after_update_job () RETURNS trigger AS $$
BEGIN
    if (NEW.status = OLD.status) then
        RETURN NEW;
    end if;

    if (NEW.status = 'completed') then
        INSERT INTO job_counters_queue
                    (task, status, count_change) VALUES
                    (OLD.task, OLD.status, -1);
        RETURN NEW;
    end if;

    if (OLD.status = 'completed') then
        INSERT INTO job_counters_queue
                    (task, status, count_change) VALUES
                    (NEW.task, NEW.status, 1);
        RETURN NEW;
    end if;

    INSERT INTO job_counters_queue
            (task, status, count_change) VALUES
            (NEW.task, NEW.status, 1),
            (OLD.task, OLD.status, -1);
    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION trigger_after_delete_job () RETURNS trigger AS $$
BEGIN
    if (OLD.status != 'completed') then
        INSERT INTO job_counters_queue
            (task, status, count_change) VALUES
            (OLD.task, OLD.status, -1);
    end if;
    RETURN OLD;
END;
$$ LANGUAGE plpgsql;


CREATE TRIGGER job_insert
AFTER INSERT ON job
FOR EACH ROW
WHEN (NEW.status != 'completed')
EXECUTE PROCEDURE trigger_after_insert_job();

CREATE TRIGGER job_update
AFTER UPDATE OF status ON job
FOR EACH ROW
WHEN (OLD.status IS DISTINCT FROM NEW.status)
EXECUTE PROCEDURE trigger_after_update_job();

CREATE TRIGGER job_delete
AFTER DELETE ON job
FOR EACH ROW
WHEN (OLD.status != 'completed')
EXECUTE PROCEDURE trigger_after_delete_job();


CREATE OR REPLACE FUNCTION update_job_counters(
    i_limit integer DEFAULT 1000
) RETURNS SETOF job_counters
LANGUAGE sql AS $$
    WITH batch AS (
        SELECT * FROM job_counters_queue
            LIMIT 1000
            FOR UPDATE SKIP LOCKED
        ),
    deleted_rows AS (
        DELETE FROM job_counters_queue
          WHERE id IN (SELECT id FROM batch)
          RETURNING *
        ),
    summarized_change AS (
        SELECT task, status, sum(count_change) AS count_change
            FROM deleted_rows
          GROUP BY task, status
          HAVING sum(count_change) != 0
        )
    INSERT INTO job_counters
    SELECT task, status, count_change FROM summarized_change
    ON CONFLICT (task, status) DO UPDATE
    SET tasks_count = job_counters.tasks_count + excluded.tasks_count
    RETURNING *
$$;



CREATE OR REPLACE FUNCTION get_job_counters(
    i_task    TEXT          DEFAULT NULL,
    i_status  job_status    DEFAULT NULL
) RETURNS SETOF job_counters
LANGUAGE sql STABLE AS $$
    WITH aggr_data AS (
        SELECT task, status, tasks_count
            FROM job_counters
            WHERE (i_task IS NULL OR task = i_task)
            AND ((i_status IS NULL AND status != 'completed') OR status = i_status)
        ),
    queue_data AS (
        SELECT task, status, sum(count_change) AS tasks_count
            FROM job_counters_queue
            WHERE (i_task IS NULL OR task = i_task)
            AND ((i_status IS NULL AND status != 'completed') OR status = i_status)
            GROUP BY task, status
        ),
    all_data AS (
        SELECT * FROM aggr_data
        UNION ALL
        SELECT * FROM queue_data
        )
    SELECT task, status, sum(tasks_count)::bigint AS tasks_count
    FROM all_data
    GROUP BY task, status;
$$;



CREATE OR REPLACE FUNCTION check_job_counters(
    i_task    TEXT,
    i_status  job_status
) RETURNS SETOF job_counters
LANGUAGE sql STABLE AS $$
    SELECT c.task, c.status,
            o.tasks_count - c.tasks_count AS count_diff
        FROM (
            SELECT (get_job_counters(jc.task, jc.status)).*,
                    jc.tasks_count AS orig_tasks_count
                FROM job_counters jc
                WHERE (i_task IS NULL OR task = i_task)
                AND ((i_status IS NULL AND status != 'completed') OR status = i_status)
        ) c,
        LATERAL (
            SELECT
                count(*) AS tasks_count
            FROM job
            WHERE
                task = c.task
                AND status = c.status
        ) o
    WHERE
        c.tasks_count <> o.tasks_count
$$;


CREATE OR REPLACE FUNCTION repair_job_counters(
    i_task    TEXT,
    i_status  job_status
) RETURNS SETOF job_counters
LANGUAGE sql AS $$
    WITH batch AS (
        SELECT * FROM job_counters_queue
            WHERE (i_task IS NULL OR task = i_task)
            AND ((i_status IS NULL AND status != 'completed') OR status = i_status)
            FOR UPDATE NOWAIT
        ),
    need_repair AS (
        SELECT (check_job_counters(task, status)).*
            FROM job_counters
            WHERE (i_task IS NULL OR task = i_task)
            AND ((i_status IS NULL AND status != 'completed') OR status = i_status)
        ),
    deleted_rows AS (
        DELETE FROM job_counters_queue jcq
            WHERE
                id IN (
                    SELECT id
                        FROM batch
                        INNER JOIN need_repair USING (task, status)
                )
        ),
    update_counters AS (
        UPDATE job_counters AS c
            SET tasks_count = c.tasks_count + n.tasks_count
            FROM need_repair n
          WHERE c.task = n.task
            AND c.status = n.status
    )
    SELECT * FROM need_repair;
$$;
