ALTER TABLE job ADD COLUMN execution_info JSONB;

DROP FUNCTION IF EXISTS find_active_job(INTEGER, TEXT);
CREATE OR REPLACE FUNCTION find_active_job(
    i_partition INTEGER, i_active_uid TEXT
) RETURNS TABLE(
    id UUID, task TEXT, active_uid TEXT,
    parameters JSONB, create_time TIMESTAMP,
    schedule_time TIMESTAMP, priority INTEGER,
    attempt INTEGER, status job_status, worker_id TEXT,
    start_time TIMESTAMP, finish_time TIMESTAMP,
    cpu_usage SMALLINT, error_message TEXT, error_stack_trace TEXT,
    execution_info JSONB
)
LANGUAGE plpgsql STABLE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    RETURN QUERY
    SELECT job.id, job.task, job.active_uid, job.parameters, job.create_time,
        job.schedule_time, job.priority, job.attempt, job.status, job.worker_id,
        job.start_time, job.finish_time, job.cpu_usage, job.error_message, job.error_stack_trace,
        job.execution_info
    FROM job WHERE job.active_uid = i_active_uid;
END
$function$;

CREATE OR REPLACE FUNCTION insert_job_ignore_id(
    i_partition INTEGER,
    i_id UUID, i_task TEXT, i_active_uid TEXT,
    i_parameters JSONB, i_create_time TIMESTAMP,
    i_schedule_time TIMESTAMP, i_priority INTEGER,
    i_attempt INTEGER, i_status job_status, i_worker_id TEXT,
    i_start_time TIMESTAMP, i_finish_time TIMESTAMP,
    i_cpu_usage INTEGER, i_error_message TEXT, i_error_stack_trace TEXT,
    i_execution_info JSONB
) RETURNS BOOLEAN
LANGUAGE plpgsql VOLATILE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    INSERT INTO job (partition, id, task, active_uid, parameters,
        create_time, schedule_time, priority, attempt, status, worker_id,
        start_time, finish_time, cpu_usage, error_message, error_stack_trace, execution_info)
    VALUES (i_partition, i_id, i_task, i_active_uid, i_parameters,
        i_create_time, i_schedule_time, i_priority, i_attempt, i_status, i_worker_id,
        i_start_time, i_finish_time, i_cpu_usage, i_error_message, i_error_stack_trace, i_execution_info)
    ON CONFLICT (id) DO NOTHING;

    RETURN FOUND;
END
$function$;

CREATE OR REPLACE FUNCTION insert_job_keep_active(
    i_partition INTEGER,
    i_id UUID, i_task TEXT, i_active_uid TEXT,
    i_parameters JSONB, i_create_time TIMESTAMP,
    i_schedule_time TIMESTAMP, i_priority INTEGER,
    i_attempt INTEGER, i_status job_status, i_worker_id TEXT,
    i_start_time TIMESTAMP, i_finish_time TIMESTAMP,
    i_cpu_usage INTEGER, i_error_message TEXT, i_error_stack_trace TEXT,
    i_execution_info JSONB
) RETURNS TABLE (id UUID, task TEXT, schedule_time TIMESTAMP)
LANGUAGE plpgsql VOLATILE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    LOOP
        RETURN QUERY
        INSERT INTO job (partition, id, task, active_uid, parameters,
            create_time, schedule_time, priority, attempt, status, worker_id,
            start_time, finish_time, cpu_usage, error_message, error_stack_trace, execution_info)
        VALUES (i_partition, i_id, i_task, i_active_uid, i_parameters,
            i_create_time, i_schedule_time, i_priority, i_attempt, i_status, i_worker_id,
            i_start_time, i_finish_time, i_cpu_usage, i_error_message, i_error_stack_trace, i_execution_info)
        ON CONFLICT (active_uid) WHERE active_uid IS NOT NULL DO NOTHING
        RETURNING job.id, job.task, job.schedule_time;

        IF NOT FOUND THEN
            RETURN QUERY SELECT job.id, job.task, job.schedule_time FROM job WHERE active_uid = i_active_uid;
        END IF;

        EXIT WHEN FOUND;
    END LOOP;
END
$function$;

CREATE OR REPLACE FUNCTION insert_job_merge_active(
    i_partition INTEGER,
    i_id UUID, i_task TEXT, i_active_uid TEXT,
    i_parameters JSONB, i_create_time TIMESTAMP,
    i_schedule_time TIMESTAMP, i_priority INTEGER,
    i_attempt INTEGER, i_status job_status, i_worker_id TEXT,
    i_start_time TIMESTAMP, i_finish_time TIMESTAMP,
    i_cpu_usage INTEGER, i_error_message TEXT, i_error_stack_trace TEXT,
    i_execution_info JSONB
) RETURNS TABLE (id UUID, task TEXT, schedule_time TIMESTAMP)
LANGUAGE plpgsql VOLATILE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    RETURN QUERY
    INSERT INTO job (partition, id, task, active_uid, parameters,
        create_time, schedule_time, priority, attempt, status, worker_id,
        start_time, finish_time, cpu_usage, error_message, error_stack_trace, execution_info)
    VALUES (i_partition, i_id, i_task, i_active_uid, i_parameters,
        i_create_time, i_schedule_time, i_priority, i_attempt, i_status, i_worker_id,
        i_start_time, i_finish_time, i_cpu_usage, i_error_message, i_error_stack_trace, i_execution_info)
    ON CONFLICT (active_uid) WHERE active_uid IS NOT NULL DO UPDATE
        SET status = excluded.status, parameters = excluded.parameters, priority = excluded.priority,
            schedule_time = excluded.schedule_time, attempt = excluded.attempt,
            start_time = COALESCE(excluded.start_time, job.start_time),
            finish_time = COALESCE(excluded.finish_time, job.finish_time),
            worker_id = COALESCE(excluded.worker_id, job.worker_id),
            cpu_usage = COALESCE(excluded.cpu_usage, job.cpu_usage),
            error_message = COALESCE(excluded.error_message, job.error_message),
            error_stack_trace = COALESCE(excluded.error_stack_trace, job.error_stack_trace),
            execution_info = COALESCE(excluded.execution_info, job.execution_info)
    RETURNING job.id, job.task, job.schedule_time;
END
$function$;

CREATE OR REPLACE FUNCTION insert_job_merge_active_if_ready(
    i_partition INTEGER,
    i_id UUID, i_task TEXT, i_active_uid TEXT,
    i_parameters JSONB, i_create_time TIMESTAMP,
    i_schedule_time TIMESTAMP, i_priority INTEGER,
    i_attempt INTEGER, i_status job_status, i_worker_id TEXT,
    i_start_time TIMESTAMP, i_finish_time TIMESTAMP,
    i_cpu_usage INTEGER, i_error_message TEXT, i_error_stack_trace TEXT,
    i_execution_info JSONB
) RETURNS TABLE (id UUID, task TEXT, schedule_time TIMESTAMP)
LANGUAGE plpgsql VOLATILE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    LOOP
        RETURN QUERY
        INSERT INTO job (partition, id, task, active_uid, parameters,
            create_time, schedule_time, priority, attempt, status, worker_id,
            start_time, finish_time, cpu_usage, error_message, error_stack_trace, execution_info)
        VALUES (i_partition, i_id, i_task, i_active_uid, i_parameters,
            i_create_time, i_schedule_time, i_priority, i_attempt, i_status, i_worker_id,
            i_start_time, i_finish_time, i_cpu_usage, i_error_message, i_error_stack_trace, i_execution_info)
        ON CONFLICT (active_uid) WHERE active_uid IS NOT NULL DO UPDATE
            SET status = excluded.status, parameters = excluded.parameters, priority = excluded.priority,
                schedule_time = excluded.schedule_time, attempt = excluded.attempt,
                start_time = COALESCE(excluded.start_time, job.start_time),
                finish_time = COALESCE(excluded.finish_time, job.finish_time),
                worker_id = COALESCE(excluded.worker_id, job.worker_id),
                cpu_usage = COALESCE(excluded.cpu_usage, job.cpu_usage),
                error_message = COALESCE(excluded.error_message, job.error_message),
                error_stack_trace = COALESCE(excluded.error_stack_trace, job.error_stack_trace),
                execution_info = COALESCE(excluded.execution_info, job.execution_info)
            WHERE job.status = 'ready'
        RETURNING job.id, job.task, job.schedule_time;

        IF NOT FOUND THEN
            RETURN QUERY SELECT job.id, job.task, job.schedule_time FROM job WHERE active_uid = i_active_uid;
        END IF;

        EXIT WHEN FOUND;
    END LOOP;
END
$function$;

CREATE OR REPLACE FUNCTION update_job(
    i_partition INTEGER,
    i_id UUID, i_task TEXT, i_active_uid TEXT,
    i_parameters JSONB, i_create_time TIMESTAMP,
    i_schedule_time TIMESTAMP, i_priority INTEGER,
    i_attempt INTEGER, i_status job_status, i_worker_id TEXT,
    i_start_time TIMESTAMP, i_finish_time TIMESTAMP,
    i_cpu_usage INTEGER, i_error_message TEXT, i_error_stack_trace TEXT,
    i_execution_info JSONB
) RETURNS BOOLEAN
LANGUAGE plpgsql VOLATILE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    UPDATE job
    SET partition = i_partition, task = i_task, active_uid = i_active_uid, parameters = i_parameters,
        create_time = i_create_time, schedule_time = i_schedule_time, priority = i_priority, attempt = i_attempt,
        status = i_status, worker_id = i_worker_id, start_time = i_start_time, finish_time = i_finish_time,
        cpu_usage = i_cpu_usage, error_message = i_error_message, error_stack_trace = i_error_stack_trace,
        execution_info = i_execution_info
    WHERE job.id = i_id;

    RETURN FOUND;
END
$function$;

CREATE OR REPLACE FUNCTION update_job_keep_active(
    i_partition INTEGER,
    i_id UUID, i_task TEXT, i_active_uid TEXT,
    i_parameters JSONB, i_create_time TIMESTAMP,
    i_schedule_time TIMESTAMP, i_priority INTEGER,
    i_attempt INTEGER, i_status job_status, i_worker_id TEXT,
    i_start_time TIMESTAMP, i_finish_time TIMESTAMP,
    i_cpu_usage INTEGER, i_error_message TEXT, i_error_stack_trace TEXT,
    i_execution_info JSONB
) RETURNS TABLE (id UUID, task TEXT, schedule_time TIMESTAMP)
LANGUAGE plpgsql VOLATILE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    LOOP
        BEGIN
            RETURN QUERY UPDATE job
            SET partition = i_partition, task = i_task, active_uid = i_active_uid, parameters = i_parameters,
                create_time = i_create_time, schedule_time = i_schedule_time, priority = i_priority, attempt = i_attempt,
                status = i_status, worker_id = i_worker_id, start_time = i_start_time, finish_time = i_finish_time,
                cpu_usage = i_cpu_usage, error_message = i_error_message, error_stack_trace = i_error_stack_trace,
                execution_info = i_execution_info
            WHERE job.id = i_id
            RETURNING job.id, job.task, job.schedule_time;

            RETURN;
        EXCEPTION WHEN unique_violation THEN
            RETURN QUERY SELECT job.id, job.task, job.schedule_time FROM job WHERE active_uid = i_active_uid;
            IF FOUND THEN
                DELETE FROM job WHERE job.id = i_id;
                RETURN;
            END IF;
        END;
    END LOOP;
END
$function$;

CREATE OR REPLACE FUNCTION update_job_merge_active(
    i_partition INTEGER,
    i_id UUID, i_task TEXT, i_active_uid TEXT,
    i_parameters JSONB, i_create_time TIMESTAMP,
    i_schedule_time TIMESTAMP, i_priority INTEGER,
    i_attempt INTEGER, i_status job_status, i_worker_id TEXT,
    i_start_time TIMESTAMP, i_finish_time TIMESTAMP,
    i_cpu_usage INTEGER, i_error_message TEXT, i_error_stack_trace TEXT,
    i_execution_info JSONB
) RETURNS TABLE (id UUID, task TEXT, schedule_time TIMESTAMP)
LANGUAGE plpgsql VOLATILE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    LOOP
        BEGIN
            RETURN QUERY UPDATE job
            SET partition = i_partition, task = i_task, active_uid = i_active_uid, parameters = i_parameters,
                create_time = i_create_time, schedule_time = i_schedule_time, priority = i_priority, attempt = i_attempt,
                status = i_status, worker_id = i_worker_id, start_time = i_start_time, finish_time = i_finish_time,
                cpu_usage = i_cpu_usage, error_message = i_error_message, error_stack_trace = i_error_stack_trace,
                execution_info = i_execution_info
            WHERE job.id = i_id
            RETURNING job.id, job.task, job.schedule_time;

            RETURN;
        EXCEPTION WHEN unique_violation THEN
            RETURN QUERY UPDATE job
            SET status = i_status, parameters = i_parameters, priority = i_priority,
                schedule_time = i_schedule_time, attempt = i_attempt,
                start_time = COALESCE(i_start_time, job.start_time),
                finish_time = COALESCE(i_finish_time, job.finish_time),
                worker_id = COALESCE(i_worker_id, job.worker_id),
                cpu_usage = COALESCE(i_cpu_usage, job.cpu_usage),
                error_message = COALESCE(i_error_message, job.error_message),
                error_stack_trace = COALESCE(i_error_stack_trace, job.error_stack_trace),
                execution_info = COALESCE(i_execution_info, job.execution_info)
            WHERE job.active_uid = i_active_uid
            RETURNING job.id, job.task, job.schedule_time;

            IF FOUND THEN
                DELETE FROM job WHERE job.id = i_id;
                RETURN;
            END IF;
        END;
    END LOOP;
END
$function$;

CREATE OR REPLACE FUNCTION update_job_merge_active_if_ready(
    i_partition INTEGER,
    i_id UUID, i_task TEXT, i_active_uid TEXT,
    i_parameters JSONB, i_create_time TIMESTAMP,
    i_schedule_time TIMESTAMP, i_priority INTEGER,
    i_attempt INTEGER, i_status job_status, i_worker_id TEXT,
    i_start_time TIMESTAMP, i_finish_time TIMESTAMP,
    i_cpu_usage INTEGER, i_error_message TEXT, i_error_stack_trace TEXT,
    i_execution_info JSONB
) RETURNS TABLE (id UUID, task TEXT, schedule_time TIMESTAMP)
LANGUAGE plpgsql VOLATILE AS $function$
DECLARE
    v_active_job RECORD;
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    LOOP
        BEGIN
            RETURN QUERY UPDATE job
            SET partition = i_partition, task = i_task, active_uid = i_active_uid, parameters = i_parameters,
                create_time = i_create_time, schedule_time = i_schedule_time, priority = i_priority, attempt = i_attempt,
                status = i_status, worker_id = i_worker_id, start_time = i_start_time, finish_time = i_finish_time,
                cpu_usage = i_cpu_usage, error_message = i_error_message, error_stack_trace = i_error_stack_trace,
                execution_info = i_execution_info
            WHERE job.id = i_id
            RETURNING job.id, job.task, job.schedule_time;

            RETURN;
        EXCEPTION WHEN unique_violation THEN
            SELECT job.id, job.task, job.schedule_time, job.status FROM job
            WHERE active_uid = i_active_uid FOR UPDATE INTO v_active_job;

            IF FOUND THEN
                IF v_active_job.status = 'ready' THEN
                    RETURN QUERY UPDATE job
                    SET status = i_status, parameters = i_parameters, priority = i_priority,
                        schedule_time = i_schedule_time, attempt = i_attempt,
                        start_time = COALESCE(i_start_time, job.start_time),
                        finish_time = COALESCE(i_finish_time, job.finish_time),
                        worker_id = COALESCE(i_worker_id, job.worker_id),
                        cpu_usage = COALESCE(i_cpu_usage, job.cpu_usage),
                        error_message = COALESCE(i_error_message, job.error_message),
                        error_stack_trace = COALESCE(i_error_stack_trace, job.error_stack_trace),
                        execution_info = COALESCE(i_execution_info, job.execution_info)
                    WHERE job.id = v_active_job.id
                    RETURNING job.id, job.task, job.schedule_time;
                ELSE
                    RETURN QUERY SELECT v_active_job.id, v_active_job.task, v_active_job.schedule_time;
                END IF;

                DELETE FROM job WHERE job.id = i_id;
                RETURN;
            END IF;
        END;
    END LOOP;
END
$function$;

DROP FUNCTION IF EXISTS find_job_by_id(INTEGER, UUID);
CREATE OR REPLACE FUNCTION find_job_by_id(
    i_partition INTEGER, i_id UUID
) RETURNS TABLE(
    id UUID, task TEXT, active_uid TEXT,
    parameters JSONB, create_time TIMESTAMP,
    schedule_time TIMESTAMP, priority INTEGER,
    attempt INTEGER, status job_status, worker_id TEXT,
    start_time TIMESTAMP, finish_time TIMESTAMP,
    cpu_usage SMALLINT, error_message TEXT, error_stack_trace TEXT,
    execution_info JSONB
)
LANGUAGE plpgsql STABLE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    RETURN QUERY
    SELECT job.id, job.task, job.active_uid, job.parameters, job.create_time,
        job.schedule_time, job.priority, job.attempt, job.status, job.worker_id,
        job.start_time, job.finish_time, job.cpu_usage, job.error_message, job.error_stack_trace,
        job.execution_info
    FROM job WHERE job.id = i_id;
END
$function$;

CREATE OR REPLACE FUNCTION update_job_execution_info(
    i_partition INTEGER,
    i_id UUID,
    i_execution_info JSONB
) RETURNS BOOLEAN
LANGUAGE plpgsql VOLATILE AS $function$
BEGIN
    PERFORM acquire_jobs_partition(i_partition);

    UPDATE job
    SET execution_info = i_execution_info
    WHERE job.id = i_id;

    RETURN FOUND;
END
$function$;
