/* pgmigrate-encoding: utf-8 */

create type promo_code_status as enum ('active', 'blocked');

create table promo_codes (
    code text primary key not null,
    user_product_price_id uuid,
    promo_template_id uuid,
    num_activations  integer,
    remaining_activations integer,
    status promo_code_status not null,
    status_updated_at  timestamp not null,
    status_reason text,
    from_date timestamp not null,
    to_date timestamp,
    created_at timestamp not null,
    updated_at timestamp not null,

    constraint fk_promo_codes_2_user_product_prices
        foreign key(user_product_price_id)
            references user_product_prices(id)
            on delete RESTRICT,
    constraint fk_promo_codes_2_promo_templates
        foreign key(promo_template_id)
            references promo_templates(id)
            on delete RESTRICT,
    constraint promo_code_exactly_one_usage check (
        (user_product_price_id is not null and promo_template_id is null)
        or
        (user_product_price_id is null and promo_template_id is not null)
        )
);
CREATE INDEX fk_promo_codes_2_user_product_prices ON promo_codes(user_product_price_id);
CREATE INDEX fk_promo_codes_2_promo_templates ON promo_codes(promo_template_id);
comment on table promo_codes is 'active promo codes. pls refer to https://wiki.yandex-team.ru/disk/billing360/projects/promokody/';

create table promo_codes_archive (
    code text primary key
);
comment on table promo_codes_archive is 'compact archive of used promo codes. Promo codes can not repeat. ever. Even when we delete them from table with active promo_codes or migrate used promo codes from MPFS';

create table user_promo_codes(
    id uuid primary key default uuid_generate_v4() not null,
    code text not null,
    uid text not null,
    created_at timestamp not null,
    order_id uuid,
    user_promo_id uuid,

    unique (uid, code), --gives index as well
    constraint fk_user_promo_codes_2_promo_codes
        foreign key(code)
            references promo_codes(code)
            on delete RESTRICT,
    constraint fk_user_promo_codes_2_orders
        foreign key(order_id)
            references orders(id)
            on delete RESTRICT,
    constraint fk_user_promo_codes_2_user_promos
        foreign key(user_promo_id)
            references user_promos(id)
            on delete RESTRICT,
    constraint user_promo_code_exactly_one_usage check (
        (order_id is not null and user_promo_id is null)
        or
        (order_id is null and user_promo_id is not null)
        )
);
CREATE INDEX fk_user_promo_codes_2_promo_codes ON user_promo_codes(code);
CREATE INDEX fk_user_promo_codes_2_orders ON user_promo_codes(order_id);
CREATE INDEX fk_user_promo_codes_2_user_promos ON user_promo_codes(user_promo_id);
comment on table user_promo_codes is 'Acts of a user using promo codes. pls refer to https://wiki.yandex-team.ru/disk/billing360/projects/promokody/';

create or replace function clone_product_for_promocode(
    new_code text,
    base_code text,
    duration_period_text text
   ) returns uuid language plpgsql
as
$$
declare
    base_code_count numeric;
    new_code_count numeric;
    new_product_id uuid;
    new_product_period_id uuid;
    new_product_price_id uuid;
begin
    select count(1) into new_code_count from user_products where code = new_code;
    select count(1) into base_code_count from user_products where code = base_code;
    if new_code_count > 0 then
        raise 'new code % is occupied. can not create product with new product code',  new_code
            USING ERRCODE = 'unique_violation';
    end if;
    if base_code_count = 0 then
        raise 'base code % is absent. can not clone product from code',  base_code
            USING ERRCODE = 'unique_violation';
    end if;
    perform copy_products(
        base_code,
        base_code,
        new_code
        );

    select id into new_product_id from user_products where code = new_code;

    --just in case it will be copied in the future
    delete from user_product_buckets where user_product_id = new_product_id;

    update user_products
    set
        billing_type = 'free',
        allow_auto_prolong = false,
        trust_subs_retry_charging_delay = null,
        trust_subs_retry_charging_limit = null,
        trust_subs_grace_period = null
    where id = new_product_id;

    --cleanup paid prices
    delete
    from user_product_prices
    where user_product_period_id in (
        select id
        from user_product_periods
        where user_product_id = new_product_id
    );

    delete
    from user_product_periods
    where id in (
        select id
        from user_product_periods
        where user_product_id = new_product_id
    );

    insert into user_product_periods
    (user_product_id,
     period,
     code,
     trust_fiscal_title,
     created_at,
     package_name,
     start_period_discount_duration_measurement,
     start_period_discount_duration_length,
     start_period_count)
    values (new_product_id,
            duration_period_text,
            new_code || '_promo_' || duration_period_text,
            null,
            now(),
            null,
            null,
            null,
            null)
    returning id into new_product_period_id;

    insert into user_product_prices
    (created_at,
     region_id,
     price,
     currency,
     display_discount_percent,
     display_original_price,
     user_product_period_id,
     start_period_price)
    values (now(),
            10000,
            0,
            'RUB',
            null,
            null,
            new_product_period_id,
            null) returning  id into new_product_price_id;
    return new_product_price_id;
end
$$;

do
$do$
declare
    test_promo_tanker_id uuid;
    test_promo_product_price_id uuid;
    test_promo_template_id uuid;
begin
    insert into tanker_keys(created_at, project, key_set, key)
    values (now(), 'disk-ps-billing', 'promos', 'test_promo_for_promo_codes')
    on conflict(project, key_set, key) do update
        set key = excluded.key --otherwise it won't return id
    returning id into test_promo_tanker_id;

    insert into promo_templates
    (code, description, promo_name_tanker_key_id, from_date, to_date, application_area, application_type, duration,
     duration_measurement, activation_email_template_key, created_at)
    values ('test_promo_for_promocodes', 'web promo for testing of promocodes', test_promo_tanker_id,
            '2022-01-01T00:00:00.000+0300' AT TIME ZONE 'UTC',
            '2222-01-01T00:00:00.000+0300' AT TIME ZONE 'UTC',
               --rest of the params are defaulted for CHEMODAN_83006
            'per_user'::promo_application_area, 'multiple_time'::promo_application_type, 5, 'minute', null, now()
           ) returning id into test_promo_template_id;
    perform link_promo('web default line', 'test_promo_for_promocodes');
    perform create_promo_payload('test_promo_for_promocodes', 'web_tuning'::payload_type,
'{
 "text_tanker": { "tanker_project": "disk-ps-billing", "tanker_key_set": "promos", "tanker_key": "test_promo_for_promo_codes" },
 "description_tanker": { "tanker_project": "disk-ps-billing", "tanker_key_set": "promos", "tanker_key": "test_promo_for_promo_codes" },
 "landing_tanker": { "tanker_project": "disk-ps-billing", "tanker_key_set": "promos", "tanker_key": "test_promo_for_promo_codes" },
 "background": "https://yastatic.net/s3/disk/promo/april-2-2022/web-tuning/bg.jpg",
 "illustration": "https://yastatic.net/s3/disk/promo/april-2-2022/web-tuning/grid-promo.png",
 "gridPromoPaddingBottom": 10,
 "theme": "dark"
}');

    select clone_product_for_promocode(
        'mail_pro_b2c_premium1000_promocode_free',
        'mail_pro_b2c_premium1000_v20210610',
        'minute'
        ) into test_promo_product_price_id;

    insert into promo_codes(code, user_product_price_id, promo_template_id, num_activations, remaining_activations, status,
                            status_updated_at, status_reason, from_date, to_date, created_at, updated_at)
    values ('TEST_CODE_FOR_1_TB_5_MIN_44HJJCJ294',
            test_promo_product_price_id,
            null,
            100500,
            100400,
            'active',
            '2022-01-01T00:00:00.000+0300' AT TIME ZONE 'UTC',
            'for_testing_product',
            '2022-01-01T00:00:00.000+0300' AT TIME ZONE 'UTC' - interval '1 day',
            '2022-01-01T00:00:00.000+0300' AT TIME ZONE 'UTC' + interval '100 year',
            now(),
            now()
            ),
           ('TEST_CODE_FOR_FAKE_PROMO_5_MIN_3SNHFXPGJI',
            null,
            test_promo_template_id,
            100300,
            100200,
            'active',
            '2022-01-01T00:00:00.000+0300' AT TIME ZONE 'UTC',
            'for_testing_promo',
            '2022-01-01T00:00:00.000+0300' AT TIME ZONE 'UTC' - interval '1 day',
            '2022-01-01T00:00:00.000+0300' AT TIME ZONE 'UTC' + interval '100 year',
            now(),
            now());
end;
$do$
