/* pgmigrate-encoding: utf-8 */

create or replace function enum_catalog_tables() returns text[] language plpgsql
as
$$
declare
begin
    return array[
        'tanker_keys',
        'product_templates',
        'user_products_to_product_lines',
        'promo_product_lines',
        'promo_payload',
        'promo_templates',
        'user_products',
        'product_lines',
        'product_features',
        'user_product_periods',
        'user_product_prices'
        ];
end
$$;

create or replace function backup_catalog(caption text) returns void language plpgsql
as
$$
declare
    table_name text;
begin
    FOREACH table_name IN ARRAY enum_catalog_tables()
    loop
        execute 'CREATE TABLE ' || caption || table_name || ' AS TABLE ' || table_name;
    end loop;
end
$$;

create or replace function catalog_diff_with_backup(caption text) returns void language plpgsql
as
$$
declare
    table_name text;
begin
    FOREACH table_name IN ARRAY enum_catalog_tables()
    loop
        execute 'drop table if exists ' || caption || '_new_changed_diff_' || table_name;
        execute 'drop table if exists ' || caption || '_absent_diff_' || table_name;
    end loop;

    execute 'drop table if exists ' || caption || '_product_2_line_short';
    execute 'drop table if exists ' || caption || '_product_lines_diff_no_order_num';

    FOREACH table_name IN ARRAY enum_catalog_tables()
    loop
        execute 'create table ' || caption || '_absent_diff_' || table_name || ' as ' ||
                '(select * from ' || caption || table_name || ' except select * from ' || table_name || ')';
        execute 'create table ' || caption || '_new_changed_diff_' || table_name || ' as ' ||
                '(select * from ' || table_name || ' except select * from ' || caption || table_name || ')';
    end loop;
    execute 'create table ' || caption || '_product_2_line_short as (' ||
            '  select pl.description, p.code, p.is_best_offer, ppl.* ' ||
            '  from (' ||
            '    select * ' ||
            '    from user_products_to_product_lines ' ||
            '    except select * from ' || caption || 'user_products_to_product_lines' ||
            '  ) ppl' ||
            '  left join user_products p on p.id = ppl.user_product_id' ||
            '  left join product_lines pl on pl.id = ppl.product_line_id' ||
            ')';
    execute 'create table ' || caption || '_product_lines_diff_no_order_num as ' ||
            '(select id, product_set_id, created_at, selector_bean_el, description ' ||
            'from product_lines ' ||
            'except ' ||
            'select id, product_set_id, created_at, selector_bean_el, description ' ||
            'from ' || caption || 'product_lines' ||
            ')';
end
$$;

create or replace function drop_catalog_backup(caption text) returns void language plpgsql
as
$$
declare
    table_name text;
begin
    FOREACH table_name IN ARRAY enum_catalog_tables()
    loop
        execute 'drop TABLE if exists ' || caption || table_name;
        execute 'drop table if exists ' || caption || '_new_changed_diff_' || table_name;
        execute 'drop table if exists ' || caption || '_absent_diff_' || table_name;
        execute 'drop table if exists ' || caption || '_product_2_line_short';
        execute 'drop table if exists ' || caption || '_product_lines_diff_no_order_num';
    end loop;
end
$$;

create or replace function user_friendly_discounted_price(
    original_price numeric,
    discount_percent numeric,
    currency text
)  returns numeric language plpgsql
as
$$
declare
    unrounded_price numeric;
BEGIN
    unrounded_price := (1.0 - discount_percent / 100.0) * original_price;
    if currency = 'RUB' then
        return round(unrounded_price);
    end if;
    if currency = 'USD' then
        return round(unrounded_price, 2);
    end if;
    raise 'unknown currency %', currency;
end
$$;

create or replace function generate_june_promos_for_Nastya(
    new_line_description_postfix text,
    new_products_postfix text,
    discount_year boolean,
    discount_month boolean,
    discount_percent numeric,
    promo_tankers text[],
    promo_dates timestamp[]
) returns void language plpgsql
as
$$
declare
    rec record;
    prec record;
    newpcode text;
    newlinedesc text;
    discounted_periods text[] := array[]::text[];
    promo_template_code text;
    promo_line_codes text[] := array[]::text[];
BEGIN
    if discount_month then
        discounted_periods := array_append(discounted_periods, 'month');
    end if;
    if discount_year then
        discounted_periods := array_append(discounted_periods, 'year');
    end if;

    for i in 1..array_upper(promo_dates, 1)
    loop
        promo_template_code := '2022_june_promocodes_4Nastya' || new_line_description_postfix || '_' || promo_tankers[i];
        insert into promo_templates
        (code, description, promo_name_tanker_key_id, from_date, to_date, application_area, application_type, duration,
         duration_measurement, activation_email_template_key, created_at)
        values (promo_template_code,
                '2022_june_promocodes_4Nastya for CHEMODAN-84069',
                tanker_key_id('disk-ps-billing', 'promos', promo_tankers[i]),
                '2022-01-01T00:00:00.000+0300' AT TIME ZONE 'UTC',
                promo_dates[i],
                'per_user'::promo_application_area, 'multiple_time'::promo_application_type, null, null, null, now()
               );
        promo_line_codes := array_append(promo_line_codes, promo_template_code);
    end loop;

    for rec in (
        with general_lines as (
            select * from product_lines pl
            where order_num >= 0
              and order_num < 100 --500 is b2b, >= 1000 is for promo lines
              and selector_bean_el = 'productLineSelectorFactory.availableSelector()'
        )
        select pl.description dsc, pl.id plid, ps.key pskey
        from general_lines pl join product_sets ps on ps.id = pl.product_set_id
        where order_num = (select min(order_num) from general_lines plpl
                           where plpl.product_set_id = pl.product_set_id )
          and ps.key not in ('inapp_android', 'inapp_ios_disk', 'inapp_ios_mail', 'yandexmail')
    ) loop
        newlinedesc := replace(rec.dsc, 'default', 'promocode') || new_line_description_postfix;
        perform clone_line(rec.dsc, newlinedesc, 1000);
        for prec in (
            select p.code pcode, pt.code ptcode, tk.key tkey
            from user_products p
                join user_products_to_product_lines uptpl
                    on p.id = uptpl.user_product_id
                join product_templates pt
                    on p.product_template_id = pt.id
                join tanker_keys tk on p.title_tanker_key_id = tk.id
            where uptpl.product_line_id = (select id from product_lines where description = newlinedesc)
            )
        loop
            newpcode := prec.pcode || new_products_postfix || rec.pskey;

            perform clone_and_replace_product(
                newpcode,
                prec.tkey,
                prec.ptcode,
                prec.pcode,
                newlinedesc
                );

            update user_product_prices up
            set start_period_price       = user_friendly_discounted_price(
                    up.price,
                    discount_percent,
                    up.currency
                ),
                display_original_price   = up.price,
                display_discount_percent = discount_percent
            where up.user_product_period_id in (
                select id
                from user_product_periods
                where period in (select * from unnest(discounted_periods))
                  and user_product_id = (select id from user_products where code = newpcode)
            );

            update user_product_periods
            set start_period_count                         = 1,
                start_period_discount_duration_length      = 1,
                start_period_discount_duration_measurement = period::duration_measurement
            where period in (select * from unnest(discounted_periods))
              and user_product_id = (select id from user_products where code = newpcode);
        end loop;

        foreach promo_template_code in array promo_line_codes
        loop
            perform link_promo(newlinedesc, promo_template_code);
        end loop;
    end loop;
end
$$;

select drop_catalog_backup('backup_CHEMODAN_83006_');
select backup_catalog('june_promocode_4_Nastya_');

select generate_june_promos_for_Nastya(
           '_s7_airlines_promocodes',
           '_s7_airlines_promocodes_',
           true,
           false,
           10,
           array ['s7_airlines_promocodes'],
           array ['2022-07-30T23:59:59.000+0300' AT TIME ZONE 'UTC']
    );
select generate_june_promos_for_Nastya(
           '_megafon_beeline_Getblogger_promocodes',
           '_megafon_beeline_Getblogger_promocodes_',
           true,
           true,
           20,
           array ['megafon_promocodes', 'beeline_promocodes', 'Getblogger_promocodes'],
           array [
               '2022-08-31T23:59:59.000+0300' AT TIME ZONE 'UTC',
               '2022-09-15T23:59:59.000+0300' AT TIME ZONE 'UTC',
               '2022-08-31T23:59:59.000+0300' AT TIME ZONE 'UTC'
               ]
    );
select generate_june_promos_for_Nastya(
           '_mir_promocodes',
           '_mir_promocodes_',
           true,
           true,
           30,
           array ['mir_promocodes'],
           array ['2022-08-31T23:59:59.000+0300' AT TIME ZONE 'UTC']
    );

drop function generate_june_promos_for_Nastya(text, text, boolean, boolean, numeric, text[], timestamp[]);

select catalog_diff_with_backup('june_promocode_4_Nastya_');

create or replace view prp as (
select p.code p_code,
       p.billing_type p_billing_type,
       p.code_family p_code_family,

       upp.code up_code,
       upp.period up_period,
       upp.period_count up_period_count,
       upp.start_period_discount_duration_measurement up_spdisc_period,
       upp.start_period_discount_duration_length up_spdisc_period_count,

       uppr.price uppr_pricer,
       uppr.currency uppr_currency,
       uppr.display_original_price uppr_orig_price,
       uppr.display_discount_percent uppr_disc_percent,

       pl.description pl_desc,
       pl.order_num pl_order,
       pl.selector_bean_el pl_selector
from
user_products p
join user_products_to_product_lines uptpl on p.id = uptpl.user_product_id
join user_product_periods upp on p.id = upp.user_product_id
join user_product_prices uppr on upp.id = uppr.user_product_period_id
join product_lines pl on uptpl.product_line_id = pl.id
                   );
