/* pgmigrate-encoding: utf-8 */
alter table groups
    add parent_group_id uuid;

create index groups_parent_group_id_index on groups (parent_group_id);

alter table groups
    add constraint groups_groups_id_fk
        foreign key (parent_group_id) references groups
            on delete restrict;

create type group_product_type as enum ('main', 'addon');

alter table group_products
    add product_type group_product_type;

update group_products
set product_type = 'main'
where product_type is null;

alter table group_products alter column product_type set not null;


create table group_product_addons
(
    id               uuid default uuid_generate_v4() not null,
    main_product_id  uuid                            not null
        constraint group_product_addons_group_products_id_fk
            references group_products
            on delete restrict,
    addon_product_id uuid                            not null
        constraint group_product_addons_group_products_id_fk_2
            references group_products
            on delete restrict,
    enabled          boolean                         not null,
    created_at       timestamp                       not null,
    updated_at       timestamp                       not null,
    unique (main_product_id, addon_product_id)
);

create index group_product_addons_addon_product_id_index on group_product_addons (addon_product_id);
create index group_product_addons_main_product_id_index on group_product_addons (main_product_id);

alter table group_product_addons
    add constraint group_product_addons_pk
        primary key (id);


CREATE OR REPLACE FUNCTION on_group_addon_change() RETURNS TRIGGER AS
$$
BEGIN
    IF exists(select *
              from group_products gp
              where gp.id = NEW.main_product_id
                and product_type != 'main')
    THEN
        RAISE EXCEPTION 'main_product_id must refer to main product';
    END IF;
    IF exists(select *
              from group_products gp
              where gp.id = NEW.addon_product_id
                and product_type != 'addon')
    THEN
        RAISE EXCEPTION 'addon_product_id must refer to addon product';
    END IF;
    return NEW;
END;
$$ LANGUAGE plpgsql VOLATILE;

CREATE TRIGGER group_addon_type_insert_validation
    BEFORE INSERT OR UPDATE
    ON group_product_addons
    FOR EACH ROW
EXECUTE PROCEDURE on_group_addon_change();


CREATE OR REPLACE FUNCTION on_group_product_type_update() RETURNS TRIGGER AS
$$
BEGIN
    IF NEW.product_type = 'main'
    THEN
        IF exists(select *
                  from group_product_addons gpa
                  where gpa.addon_product_id = NEW.id)
        THEN
            RAISE EXCEPTION 'Product already specified as addon';
        END IF;
    END IF;
    IF NEW.product_type = 'addon'
    THEN
        IF exists(select *
                  from group_product_addons gpa
                  where gpa.main_product_id = NEW.id)
        THEN
            RAISE EXCEPTION 'Product already specified as main';
        END IF;
    END IF;
    return NEW;
END;
$$ LANGUAGE plpgsql VOLATILE;

CREATE TRIGGER on_group_product_type_update
    BEFORE UPDATE OF product_type
    ON group_products
    FOR EACH ROW
EXECUTE PROCEDURE on_group_product_type_update();

/*
-- validation queries

--expect fail
insert into  group_product_addons
(main_product_id, addon_product_id, created_at, updated_at, enabled)
values (
    (select id from group_products where code = 'org_mail_pro_standard100'),
    (select id from group_products where code = 'org_mail_pro_standard100'),
        now(),
        now(),
        true);

--expect ok
update group_products set product_type = 'addon' where code = 'org_mail_pro_standard100';
--expect fail
insert into  group_product_addons
(main_product_id, addon_product_id, created_at, updated_at, enabled)
values (
           (select id from group_products where code = 'org_mail_pro_standard100'),
           (select id from group_products where code = 'org_mail_pro_standard200'),
           now(),
           now(),
           true);

--expect ok
insert into  group_product_addons
(main_product_id, addon_product_id, created_at, updated_at, enabled)
values (
           (select id from group_products where code = 'org_mail_pro_standard200'),
           (select id from group_products where code = 'org_mail_pro_standard100'),
           now(),
           now(),
           true);

--expect fail
update group_products set product_type = 'main' where code = 'org_mail_pro_standard100';
*/
