#!/usr/bin/env python

import psycopg2
import time
import sys
import socket

def get_hostname():
    hostname = socket.getfqdn()
    hostname = hostname.replace('.', '_')
    return hostname.replace('-', '_')

def remove_unused_chars(touple=('', '', '')):
    (dbid, app, size) = touple
    for c in '.@$-':
        if len(app) == 0:
            app = 'empty_string'
        if len(dbid) == 0:
            dbid = 'empty_string'
        app = app.replace(c, '_')
        app = app.lstrip('_')
        dbid = dbid.replace(c, '_')
        dbid = dbid.lstrip('_')
    return dbid, app, size

def send_to_graphite(res=[]):
    hostname = get_hostname()
    timestamp = int(time.time())
    prefix = 'disk.pg.%s.db.apidb.p_data_size' % hostname

    s = socket.create_connection(("localhost", 42000), 60)
    for touple in res:
        (dbid, app, size) = remove_unused_chars(touple)
        result = '%s.%s.%s %s %s\n' % (prefix, dbid, app, str(int(size)),\
            timestamp)
        s.send(result.encode('utf-8'))

def main():
    conn = psycopg2.connect('dbname=api_disk_data')
    conn.autocommit = True
    cur = conn.cursor()

    cur.execute("SHOW transaction_read_only")
    if cur.fetchone()[0] != 'off':
        sys.exit(0)

    cur.execute("SELECT tablename"
                "    FROM pg_tables"
                "  WHERE schemaname='public'"
                "    AND tablename ~ 'p_data_'"
                "  ORDER BY tablename")
    tables = [i[0] for i in cur.fetchall()]

    cur.execute("CREATE TEMPORARY TABLE IF NOT EXISTS apidb_data_size"
                "    (app text, dbid text, chunk_size bigint)")

    for table in tables:
        cur.execute("INSERT INTO apidb_data_size"
                    "    SELECT coalesce(app, 'other') AS app,"
                    "           coalesce(dbid, 'other') AS dbid,"
                    "           sum(pg_column_size(t.*))"
                    "        FROM %s t"
                    "      GROUP BY app, dbid" % table)

    cur.execute("WITH top AS ("
                "    SELECT dbid, app, sum(chunk_size)"
                "        FROM apidb_data_size"
                "      GROUP BY dbid, app"
                "      ORDER BY sum(chunk_size) DESC"
                "      LIMIT 5"
                "    ), "
                "other AS ("
                "    SELECT 'other'::text, 'other'::text, sum(chunk_size)"
                "        FROM apidb_data_size"
                "      WHERE (dbid, app) NOT IN ("
                "          SELECT dbid, app FROM top)"
                "    )"
                "SELECT * FROM top "
                "UNION ALL "
                "SELECT * FROM other")
    res = cur.fetchall()

    conn.close()
    send_to_graphite(res)

if __name__ == '__main__':
    main()
