#!/usr/bin/env python

import sys
import threading
import time

import psycopg2
from contextlib import closing

LIMIT = {{salt['pillar.get']('data:update_job_counters_limit', '50000')}}
THREADS = {{salt['pillar.get']('data:update_job_counters_threads', '1')}}


def main():
    exit_on_slave()

    if THREADS == 1:
        print('Threads: %s, no threads needed' % THREADS)
        update_counters("None")
    else:
        print('Threads: %s, starting threads' % THREADS)
        for index in range(THREADS):
            x = threading.Thread(target=update_counters, args=(index,))
            x.start()


def exit_on_slave():
    with closing(psycopg2.connect("dbname=diskqdb")) as conn:
        with conn.cursor() as cur:
            cur.execute("SELECT pg_is_in_recovery()")
            if cur.fetchone()[0]:
                print("Nothing to do on replica. Exiting")
                sys.exit(0)


def update_counters(name):
    with closing(psycopg2.connect("dbname=diskqdb")) as conn:
        conn.autocommit = True
        with conn.cursor() as cur:
            cur.execute('SET enable_nestloop=off')
            while True:
                sql = '''
                    WITH deleted_rows AS (
                        DELETE FROM job_counters_queue j
                            WHERE j.id in (
                                SELECT id
                                FROM job_counters_queue
                                LIMIT %s
                                FOR UPDATE SKIP LOCKED
                            ) RETURNING j.task, j.status, j.count_change
                    ),
                    summarized_change AS (
                             SELECT task, status, sum(count_change) AS count_change
                             FROM deleted_rows
                             GROUP BY task, status
                             HAVING sum(count_change) != 0
                             ORDER BY task, status
                         ), 
                    upsert as (
                              INSERT INTO job_counters
                                SELECT task, status, count_change FROM summarized_change
                                ON CONFLICT (task, status) DO UPDATE
                                    SET tasks_count = job_counters.tasks_count + excluded.tasks_count
                        )
                    select count(*) from deleted_rows'''
                cur.execute(sql, (LIMIT,))
                cnt = cur.fetchone()[0]
                print('Thread %s deletes %d rows from job_counters_queue during this run' % (name, cnt))
                if not cnt == LIMIT:
                    time.sleep(1)


if __name__ == '__main__':
    main()
