#!/usr/bin/python

import psycopg2
import subprocess
import sys
import logging
import multiprocessing
import time


class Database():
    def __init__(self, dbname, schema, table):
        try:
            global log
            self.log = log
            self.dbname = dbname
            self.schema = schema
            self.table = table
            self.tablespace = 'sata'
            self.conn = psycopg2.connect('dbname=%s user=postgres connect_timeout=1' % self.dbname)
            self.conn.autocommit = True
            self.pg_version = self.get_pg_version()
            if not self.get_pg_role():
                self.log.info("Not a master host. Exiting.")
                sys.exit(0)
        except psycopg2.OperationalError:
            self.log.error('Could not connect to postgres. Exiting')
            sys.exit(1)

    def get_pg_version(self):
        try:
            cur = self.conn.cursor()
            cur.execute("SELECT regexp_matches(version(), E'PostgreSQL ([0-9]*.[0-9]*)')")
            pg_version = str(cur.fetchone()[0][0])
            cur.close()
            return pg_version
        except Exception as err:
            self.log.error(str(err))
            sys.exit(1)

    def get_pg_role(self):
        try:
            cur = self.conn.cursor()
            cur.execute("SELECT not (setting)::boolean FROM pg_settings WHERE name = 'transaction_read_only';")
            res = cur.fetchone()[0]
            cur.close()
            return res
        except Exception as err:
            self.log.error(str(err))
            sys.exit(1)

    def get_list_of_unrepacked_tables(self):
        try:
            cur = self.conn.cursor()
            cur.execute("SELECT relname \
                    FROM pg_class \
                    WHERE relname IN (\
                        SELECT tablename FROM pg_catalog.pg_tables \
                            WHERE schemaname='%s' AND tablename like '%s_p%%' \
                        )\
                        AND reltablespace = 0 \
                        AND relname < concat('%s_p', \
                            to_char(current_timestamp - '5 days'::interval, 'YYYY_MM_DD'))\
                    ORDER BY relname;" % (self.schema, self.table, self.table))
            res = cur.fetchall()
            cur.close()
            return [i[0] for i in res]
        except Exception as err:
            self.log.error(str(err))
            sys.exit(1)

    def get_list_of_local_pids(self):
        try:
            cur = self.conn.cursor()
            cur.execute("SELECT pid, query FROM pg_stat_activity WHERE usename='postgres' AND \
                    (query LIKE 'CREATE%' OR query LIKE 'autovacuum%(to prevent wraparound)');")
            res = cur.fetchall()
            cur.close()
            for i in res:
                log.debug('Found pid ' + str(i[0]) + ' with query ' + str(i[1]))
            return [i[0] for i in res]
        except Exception as err:
            self.log.error(str(err))
            sys.exit(1)

    def repack_table(self, tablename):
        try:
            cmd = "/usr/pgsql-%s/bin/pg_repack -e -t %s.%s -s %s -S %s" % \
                (self.pg_version, self.schema, tablename, self.tablespace, self.dbname)
            log.info(cmd)
            p = subprocess.Popen(cmd, shell=True, bufsize=-1, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            while p.poll() is None:
                line = p.stderr.readline()
                if line:
                    print log.info('repack: ' + line.rstrip())
            if p.returncode != 0:
                self.log.error("Error while using pg_repack. Exiting.")
                for line in p.stderr.readlines():
                    log.error('repack: ' + line.rstrip())
                sys.exit(1)
            log.info("Moving %s to %s tablespace successfully finished." % (tablename, self.tablespace))
        except Exception as err:
            self.log.error(str(err))
            sys.exit(1)

def parse_args(argv):
    from optparse import OptionParser
    parser = OptionParser()
    parser.add_option("--log-level", help="[ %default ] info warning error", default="debug", action="store", dest="log_level")
    parser.add_option("--dbname", help="Name of the database where pg_repack is installed (mandatory)", action="store", dest="dbname")
    parser.add_option("--schema", help="Schema name where tables for repacking are (mandatory)", action="store", dest="schema")
    parser.add_option("--table", help="Tablename template to repack (mandatory)", action="store", dest="table")
    parser.add_option("--group", help="Cgroup for processes limiting (mandatory)", action="store", dest="group")
    (options, arguments) = parser.parse_args(argv)
    if (len(arguments)>1):
        raise Exception("Tool takes no arguments.")
    return options

def init_logging():
    global log
    log = logging.getLogger('repacking')
    log.setLevel(10)
    _format = logging.Formatter("%(levelname)s\t%(asctime)s\t\t%(message)s")
    _handler = logging.FileHandler("/tmp/repacking.log")
    _handler.setFormatter(_format)
    _handler.setLevel(10)
    log.addHandler(_handler)

def repacking(options, db):
    global log
    log.info("Process for repacking started.")
    tables = db.get_list_of_unrepacked_tables()
    if len(tables) == 0:
        log.info("No tables found to repack. Exiting.")
        sys.exit(0)
    for table in tables:
        db.repack_table(table)
    log.info("Process for repacking finished.")

def cgroup_adding(options, db):
    global log
    log.info("Process for adding local pg backends to cgroup started.")
    while True:
        time.sleep(10)
        pids = db.get_list_of_local_pids()
        log.debug(pids)
        if len(pids) == 0:
            continue
        subprocess.call(["cgclassify", "-g", "blkio:" + options.group] + map(str, pids))

if __name__ == '__main__':
    options = parse_args(sys.argv)
    init_logging()
    db = Database(options.dbname, options.schema, options.table)

    p_main = multiprocessing.Process(target=repacking, name='repacking', args=(options, db))
    p_main.start()

    p_cgroup = multiprocessing.Process(target=cgroup_adding, name='cgroup adding', args=(options, db))
    p_cgroup.daemon = True
    p_cgroup.start()

    p_main.join()
    if p_cgroup.is_alive():
        p_cgroup.terminate()
        log.info("Process for adding local pg backends to cgroup finished.")
