#!/usr/bin/env python

import psycopg2
import sys
import threading
import time


DBNAME = '{{ salt['pillar.get']('data:pg_repack:dbname', 'xivadb') }}'
SCHEMA = '{{ salt['pillar.get']('data:pg_repack:schema', 'xiva') }}'
lock = threading.RLock()


def get_table_list(conn):
    table = '{{ salt['pillar.get']('data:pg_repack:table', 'notifications') }}'
    retention_period = '{{ salt['pillar.get']('data:pg_repack:retention_period', '4') }}'
    result = []
    query = """SELECT relname FROM pg_class
        WHERE relname
        IN (
            SELECT
                tablename
            FROM
                pg_catalog.pg_tables
            WHERE
                schemaname='%s' AND tablename ~ '%s_p'
        )
        AND
        relname < concat('%s_p', to_char(current_timestamp - '%s days'::interval, 'YYYY_MM_DD'))
        ORDER BY
            relname;""" % (SCHEMA, table, table, retention_period)
    try:
        cur = conn.cursor()
        cur.execute(query)
        result = [row[0] for row in cur.fetchall()]
    except:
        result = []

    return result


def maintenance(conn, table):
    cur = conn.cursor()
    lock.acquire(blocking=0)
    try:
        cur.execute("DROP TABLE %s.%s;" % (SCHEMA, table))
    except psycopg2.Warning as e:
        print("WARNING: %s" % str(e))
    except psycopg2.Error as e:
        print("ERROR: %s" % str(e))
    finally:
        lock.release()
        cur.close()


def terminator(conn, pid, table):
    kill_query = u"""SELECT
                       waiting.locktype           AS waiting_locktype,
                       waiting.relation::regclass AS waiting_table,
                       waiting_stm.query          AS waiting_query,
                       waiting.mode               AS waiting_mode,
                       waiting.pid                AS waiting_pid,
                       other.locktype             AS other_locktype,
                       other.relation::regclass   AS other_table,
                       other_stm.query            AS other_query,
                       other.mode                 AS other_mode,
                       other.pid                  AS other_pid,
                       other.granted              AS other_granted,
                       pg_terminate_backend(other.pid)
                   FROM
                       pg_catalog.pg_locks AS waiting
                   JOIN
                       pg_catalog.pg_stat_activity AS waiting_stm
                       ON (
                           waiting_stm.pid = waiting.pid
                       )
                   JOIN
                       pg_catalog.pg_locks AS other
                       ON (
                           (
                               waiting."database" = other."database"
                           AND waiting.relation  = other.relation
                           )
                           OR waiting.transactionid = other.transactionid
                       )
                   JOIN
                       pg_catalog.pg_stat_activity AS other_stm
                       ON (
                           other_stm.pid = other.pid
                       )
                   WHERE
                       NOT waiting.granted
                   AND
                       waiting.pid <> other.pid and waiting.pid=%(pid)s""";
    check_query = """SELECT tablename FROM 
                        pg_tables 
                    WHERE
                        schemaname=%s
                    AND
                        tablename=%s"""
    counter = 0
    try:
        cur = conn.cursor()
        cur.execute("SET temp_file_limit = '10MB'")
        while not lock.acquire(blocking=0) and counter <= 10:
            cur.execute(kill_query, {'pid': pid})
            for killed in cur.fetchall():
                print killed 

            counter += 1
            time.sleep(0.1)
    except (psycopg2.Warning, psycopg2.Error):
        pass
    finally:
        lock.release()
        cur.close()


def main():
    try:
        conn = psycopg2.connect('dbname=postgres connect_timeout=1')
        conn.autocommit = True
        cur = conn.cursor()

        cur.execute("SHOW transaction_read_only;")
        if 'on' in str(cur.fetchone()[0]):
            sys.exit(1)

        maintenance_conn = psycopg2.connect('dbname=%s connect_timeout=1' % DBNAME)
        maintenance_conn.autocommit = True
        m_cur = maintenance_conn.cursor()
        m_cur.execute('SELECT * FROM pg_backend_pid()')
        pid = m_cur.fetchone()[0]
    except (psycopg2.Error, psycopg2.Warning):
        print 'Cannot connect or get backend pid'
        m_cur.close()
        maintenance_conn.close()
        cur.close()
        conn.close() 
        sys.exit(1)

    for table in get_table_list(maintenance_conn):
        tm = threading.Thread(None, maintenance, 'maintenance', (maintenance_conn, table))
        tt = threading.Thread(None, terminator, 'terminator', (conn, pid, table))
        tm.start()
        tt.start()
        tm.join()
        tt.join()

    m_cur.close()
    maintenance_conn.close()
    cur.close()
    conn.close()


if __name__ == '__main__':
    main()
