#!/usr/bin/python
import psycopg2
import sys
import logging
import subprocess
import os

class Database():
    def __init__(self, dbname, schema, table, group):
        try:
            self.retention_period = {{ salt['pillar.get']('data:pg_repack:retention_period', '4') }}
            self.res_length_limit = self.retention_period #- 1 
            if self.res_length_limit < 0:
                self.res_length_limit = 0
            self.log = log
            self.dbname = dbname
            self.schema = schema
            self.table = table + '_p%'
            self.group = group
            self.fl = False
            self.children_table = 'partman.frozenvacuum_processed_children'
            self.conn = psycopg2.connect('dbname=%s connect_timeout=1' % self.dbname )
            self.conn.autocommit = True

        except psycopg2.OperationalError:
            self.log.error('Could not connect to postgres. Exiting')
            sys.exit(1)

    def _exec_query(self, query):
        res = ''
        try:
            cur = self.conn.cursor()
            cur.execute(query)
        except Exception as err:
            self.log.error(str(err))
            sys.exit(1)

        try:
            res = cur.fetchall()
        except psycopg2.ProgrammingError as err:
            err_str=str(err)
            if 'no results to fetch' not in err_str: 
                self.log.error(str(err)) 
        finally:
            cur.close()

        return res

    def apply_io_limit(self):
        res = self._exec_query('SELECT pg_backend_pid();') 
        pid = str(res[0][0]) 
        subprocess.call(["cgclassify", "-g", "blkio:{0}".format(self.group), pid])

    def call_frozen_vacuum(self, tblname):
        self._exec_query('CREATE TABLE IF NOT EXISTS %s(name varchar(30) primary key,\
 start_ts timestamptz, stop_ts timestamptz);' % self.children_table)
        raw_set = self._exec_query(\
'SELECT name FROM %s WHERE stop_ts IS NOT NULL ORDER BY name ASC;' % self.children_table)
        names_set = [n[0] for n in raw_set]
        table = "%s.%s" % (self.schema,tblname)
        if table not in names_set:
             log.info("Frozenvacuumizer: start vacuum on a %s" % table)
             _u = self._exec_query('INSERT INTO %s (name,start_ts) VALUES ($$%s$$,now()) \
ON CONFLICT (name) DO UPDATE SET start_ts = now();' %(self.children_table,table))
             _u = self._exec_query('VACUUM FREEZE %s;' % table)
             log.info("Frozenvacuumizer: Finished vacuum on the table %s" % table)
             _u = self._exec_query('UPDATE %s SET stop_ts=now() WHERE name=$$%s$$;'\
%(self.children_table,table))
        else:
             log.info("Frozenvacuum: The child %s has been already processed." % table)

    def clean_children_table(self):
       	self._exec_query('DELETE FROM %s WHERE stop_ts<now()-$$15 days$$::interval;' % self.children_table)

    def get_list_of_child_partitions(self):
        tbllist = []
        res = self._exec_query("SELECT relname FROM pg_class \
            WHERE relname IN (\
            SELECT tablename FROM pg_catalog.pg_tables \
                WHERE schemaname='%s' AND tablename LIKE '%s') \
                AND relname < concat('%s', to_char(current_timestamp, 'YYYY_MM_DD')) \
            ORDER BY relname;" % (self.schema, self.table, self.table)
       )
        res = [ t[0] for t in res ]

        if len(res) > 0:
            tbllist = res
        else:
            log.info("Frozenvacuum: No tables to vacuumize yet.")

        return tbllist 


def parse_args():
    from optparse import OptionParser
    parser = OptionParser()
    parser.add_option("--log-level", help="[ %default ] info warning error", default="debug", action="store", dest="log_level")
    parser.add_option("--group", help="Cgroup for io limiting", default="frozen_vacuum", action="store", dest="group")
    parser.add_option("--dbname", help="Name of the database where partitioned table locates (mandatory)", action="store", dest="dbname")
    parser.add_option("--schema", help="Schema name where partitioned table locates (mandatory)", action="store", dest="schema")
    parser.add_option("--table", help="A basename for a partitioned table (mandatory)", action="store", dest="table")

    (options, arguments) = parser.parse_args(sys.argv)
    if None in (options.dbname, options.schema, options.table):
        parser.print_usage()
        sys.exit(1)
    if (len(arguments)>1):
        raise Exception("Tool takes no arguments.")
    return options

def init_logging():
    global log
    log = logging.getLogger('frozenvacuum')
    log.setLevel(10)
    _format = logging.Formatter("%(levelname)s\t%(asctime)s\t\t%(message)s")
    _handler = logging.FileHandler("/var/log/postgresql-frozenvacuum/frozenvacuum.log")
    _handler.setFormatter(_format)
    _handler.setLevel(10)
    log.addHandler(_handler)

if __name__ == '__main__':
    init_logging()
    options = parse_args()
    db = Database(options.dbname, options.schema, options.table, options.group)
    log.info("The proccess of frozenvacuum on a partitioned table %s in a db %s has started" % (db.table, db.dbname))
    tbllist=db.get_list_of_child_partitions()
    db.apply_io_limit()
    for table in tbllist:
        db.call_frozen_vacuum(table)
    db.clean_children_table()
    log.info("The proccess of frozenvacuum on a partitioned table %s in a db %s has finished" % (db.table, db.dbname)) 

