#!/bin/bash

WAL_COMPRESSION={{ salt['pillar.get']('data:backup:archive:wal_compression', '1') }}
MAX_PROCESSES={{ salt['pillar.get']('data:backup:archive:max_processes', '4') }}
MAX_FILES_IN_CHUNK={{ salt['pillar.get']('data:backup:archive:max_files_in_chunk', '64') }}
MAX_FILES_TOTAL={{ salt['pillar.get']('data:backup:archive:max_files_total', '1024') }}
TMPDIR="{{ salt['pillar.get']('data:backup:archive:tmpdif', '/dev/shm/wals') }}"
{% if salt['grains.get']('os') == 'Ubuntu' %}
{% set rename_cmd = "rename 's/\.gz//'" %}
{% else %}
{% set rename_cmd = "rename '.gz' ''" %}
{% endif %}
WALSDIR="{{ salt['pillar.get']('data:backup:archive:walsdir', pg_prefix + '/wals') }}"
{% set barman_server = salt['pillar.get']('data:backup:server') %}
{% set short_name = salt['pillar.get']('data:backup:short_name', salt['grains.get']('pg:suffix')) %}
RSYNC_SERVER="{{ barman_server }}"
RSYNC_PATH="/barman/{{ short_name }}/incoming"
RSYNC_BWLIMIT={{ salt['pillar.get']('data:backup:archive:bwlimit', '81920') }} # 80 MB/s by default
RSYNC_CMD="rsync -a --timeout=60 --contimeout=1 --password-file=/etc/barman.passwd"
EXPECTED_WAL_SIZE={{ salt['pillar.get']('data:backup:archive:walsize', '16777216') }}

echo -e "$(date +%F\ %T)\tStarting new interation"

if [ "$WAL_COMPRESSION" -eq 0 ]; then
    RSYNC_CMD="$RSYNC_CMD -L"
fi

if [ ! -d "$TMPDIR" ]; then
    mkdir -p $TMPDIR
fi

# Cleanup from previous run
if [ $(ls -1 $TMPDIR/ | wc -l) -ne 0 ]; then
    echo -e "$(date +%F\ %T)\tCleaning up from previous run"
    echo -e "$(date +%F\ %T)\t$(ls -l $TMPDIR/)"
    if [ "$WAL_COMPRESSION" -ne 0 ]; then
        find $TMPDIR -type l -delete || exit 50
        find $TMPDIR -type f -name '*.gz' -delete || exit 60
    fi
    # Here we do it one by one since in previous run we already tried
    # to rsync all files with one exec
    for file in $(ls -1 $TMPDIR/); do
        file_size=$(stat -c %s ${TMPDIR}/${file})
        echo -e "$(date +%F\ %T)\tRsyncing ${file} of size ${file_size} left from previous run"
        ${RSYNC_CMD} --timeout=5 $TMPDIR/$file \
            rsync://barman@${RSYNC_SERVER}:${RSYNC_PATH}/ && \
            rm -f $TMPDIR/$file && rm -f $WALSDIR/$file || exit 70
    done
fi

processed=0
while [ "$processed" -lt "$MAX_FILES_TOTAL" ]
do
    chunk=`ls -1 ${WALSDIR} 2>/dev/null | head -n ${MAX_FILES_IN_CHUNK}`
    echo -e "$(date +%F\ %T)\tChunk files are:\n${chunk}"
    if [ -z "$chunk" ]; then
        echo -e "$(date +%F\ %T)\tCompleted iteration (no more WALs available)"
        exit 0
    fi

    for wal in ${chunk}; do
        wal_size=$(stat -c %s ${WALSDIR}/${wal})
        echo -e "$(date +%F\ %T)\tProcessing WAL ${wal} of size ${wal_size}"
        is_history=$(echo ${wal} | grep -c -E '^[0-9A-F]{8}\.history$')
        is_backup=$(echo ${wal} | grep -c -E '^[0-9A-F]{24}\.[0-9A-F]{8}\.backup$')
        is_utility=$((is_history + is_backup))
        if [ ${wal_size} -ne $EXPECTED_WAL_SIZE -a ${is_utility} -eq 0 ]; then
            echo "Unexpected size for WAL ${wal}: ${wal_size} bytes"
            exit 15
        fi
        ln -s -f ${WALSDIR}/${wal} ${TMPDIR}/${wal} || exit 10
    done

    if [ "$WAL_COMPRESSION" -ne 0 ]; then
        echo -e "$(date +%F\ %T)\tCompressing WALs"
        pigz -p ${MAX_PROCESSES} -f ${TMPDIR}/* || exit 20
        echo -e "$(date +%F\ %T)\tRenaming WALs"
        {{ rename_cmd }} ${TMPDIR}/*.gz || exit 30
    fi

    echo -e "$(date +%F\ %T)\tRunning rsync"
    ${RSYNC_CMD} --bwlimit=${RSYNC_BWLIMIT} ${TMPDIR}/* \
        rsync://barman@${RSYNC_SERVER}:${RSYNC_PATH}/ || exit 40
    echo -e "$(date +%F\ %T)\tRemoving chunk WALs from ${TMPDIR}"
    cd ${TMPDIR} && rm -f ${chunk}
    echo -e "$(date +%F\ %T)\tRemoving chunk WALs from ${WALSDIR}"
    cd ${WALSDIR} && rm -f ${chunk}

    processed=$((processed+MAX_FILES_IN_CHUNK))
    echo -e "$(date +%F\ %T)\tProcessed ${processed} WALs during this run"
done
echo -e "$(date +%F\ %T)\tCompleted iteration (max WALs count reached)"
