#!/usr/bin/env python
# coding: utf-8

from __future__ import print_function

from argparse import ArgumentParser
from datetime import datetime
import logging
import subprocess
import threading
import time
import os.path
import json

import psycopg2


log = logging.getLogger(__name__)


class Posgres(object):
    """
    Postgres helpers
    """
    restore_port = 3432

    def __init__(self, pg_bin, pg_data, pg_config):
        self.bin = pg_bin
        self.data = pg_data
        self.config = pg_config

    def make_pg_cmd(self, cmd, *cmd_args):
        return [
            os.path.join(self.bin, cmd)
        ] + list(cmd_args)

    @classmethod
    def make_restore_dsn(cls):
        return 'dbname=postgres port=%d' % cls.restore_port


def run_command(cmd_args):
    log.info('Execute %s', cmd_args)
    cmd = subprocess.Popen(
        cmd_args, stderr=subprocess.PIPE, stdout=subprocess.PIPE,
    )
    cmd.wait()

    if cmd.returncode is not None:
        log.info('Out: %s', cmd.stdout.read())
    err_out = cmd.stderr.read().strip()
    if err_out:
        log.warning('Err: %s', err_out)
    log.log(
        logging.INFO if cmd.returncode == 0 else logging.WARNING,
        'command exit with %r code', cmd.returncode
    )


def run_restore(pg, max_connections):
    postgre_options = [
        '-p', str(pg.restore_port),
        '-c', 'config_file=%s' % pg.config,
        '-c', 'archive_mode=off',
        '-c', 'max_connections=%d' % max_connections,
        '-c', 'synchronous_commit=local',
    ]
    cmd_args = pg.make_pg_cmd(
        'pg_ctl',
        'start',
        '-D', pg.data,
        '-w',  # wait until operation completes
        '-o', ' '.join(postgre_options),
    )
    run_command(cmd_args)


def stop_restored_postgres(pg):
    cmd_args = pg.make_pg_cmd(
        'pg_ctl',
        'stop',
        '-D', pg.data,
        '-w',
    )
    run_command(cmd_args)


def is_postgres_ready(pg):
    try:
        conn = psycopg2.connect(pg.make_restore_dsn())
        log.debug('Connected to postgres')
    except psycopg2.OperationalError as exc:
        log.debug('Can\'t connect to postgres: %s', exc)
        return False

    conn.autocommit = True
    cur = conn.cursor()
    cur.execute('SELECT pg_is_in_recovery()')
    is_in_recovery = cur.fetchone()[0]

    log.debug('is_in_recovery: %r', is_in_recovery)
    conn.close()
    return not is_in_recovery


WORKAROUNDS_QUERIES = [
    'ALTER SYSTEM RESET synchronous_standby_names',
    'SELECT pg_reload_conf()',
]

RENAME_Q_TMPL = 'ALTER DATABASE {from_db_name} RENAME TO {to_db_name}'


def init_logging():
    root_logger = logging.getLogger()
    formatter = logging.Formatter(
        '%(asctime)s - %(funcName)s [%(levelname)s]: %(message)s'
    )
    root_logger.setLevel(logging.DEBUG)

    to_file = logging.FileHandler(
        '/var/log/postgresql/restore-from-backup.log')
    to_file.setFormatter(formatter)
    to_file.setLevel(logging.DEBUG)
    root_logger.addHandler(
        to_file
    )

    to_out = logging.StreamHandler()
    to_out.setFormatter(formatter)
    to_out.setLevel(logging.WARNING)
    root_logger.addHandler(
        to_out
    )


class RestoreActions(object):
    """
    restore action
    """

    log_restoring_seconds = 180
    sleep_while_wait_for_resting = 2.5

    def __init__(
            self, state_dir,
            rename_databases_from_to,
            pg, source_max_connections):
        self.state_dir = state_dir
        self.rename_databases_from_to = rename_databases_from_to
        self.pg = pg
        self.source_max_connections = source_max_connections
        self.restore_thread = None

    def start_postgres(self):
        """
        Start postgres and wait until it ready
        """
        self.restore_thread = threading.Thread(
            target=run_restore,
            kwargs=dict(
                pg=self.pg,
                max_connections=self.source_max_connections))

        self.restore_thread.start()
        log.debug('Restore threads started')
        log_restoted_at = 0
        while True:
            if not self.restore_thread.is_alive():
                raise RuntimeError('Restore thread is dead!')
            if is_postgres_ready(self.pg):
                log.info('Postgres ready')
                return
            if (time.time() - log_restoted_at) > self.log_restoring_seconds:
                log.debug('Postgres restoring')
                log_restoted_at = time.time()
            time.sleep(self.sleep_while_wait_for_resting)

    def apply_workarounds(self):
        conn = psycopg2.connect(self.pg.make_restore_dsn())
        conn.autocommit = True
        for q in WORKAROUNDS_QUERIES:
            log.info('Apply %s', q)
            cur = conn.cursor()
            cur.execute(q)
        for from_db_name, to_db_name in self.rename_databases_from_to.items():
            log.info('Rename %s to %s', from_db_name, to_db_name)
            cur = conn.cursor()
            cur.execute(
                RENAME_Q_TMPL.format(
                    from_db_name=from_db_name,
                    to_db_name=to_db_name)
            )
        conn.close()

    def stop_postgres(self):
        stop_restored_postgres(self.pg)
        if self.restore_thread is not None:
            self.restore_thread.join()

    def _path_to(self, name):
        return os.path.join(self.state_dir, name.__name__)

    def _state_passed(self, name):
        return os.path.exists(self._path_to(name))

    def _mark_state_passed(self, name):
        with open(self._path_to(name), 'w') as fd:
            fd.write('done at %s' % datetime.now())

    def __call__(self):
        for func in [
                self.start_postgres,
                self.apply_workarounds,
                self.stop_postgres]:
            if self._state_passed(func):
                log.info('Skip state %r, cause it completed', func)
                continue
            log.info('Start step %r', func)
            func()
            self._mark_state_passed(func)
            log.info('Step %r finished', func)


def main():
    parser = ArgumentParser()
    parser.add_argument(
        '--recovery-state',
        metavar='PATH',
        help='path to recovery state dir',
    )
    parser.add_argument(
        '--pg-data',
        metavar='PATH',
        help='path to pg.data',
        required=True,
    )
    parser.add_argument(
        '--pg-bin',
        metavar='PATH',
        help='path to postgres utils',
        required=True,
    )
    parser.add_argument(
        '--pg-config',
        metavar='FILE',
        help='path to postgres config',
        required=True,
    )
    parser.add_argument(
        '--rename-databases-from-to',
        metavar='JSON',
        help='json dict',
        required=True,
    )
    parser.add_argument(
        '--source-max-connections',
        type=int,
        required=True,
    )

    init_logging()
    args = parser.parse_args()
    rename_databases_from_to = json.loads(args.rename_databases_from_to)

    restore = RestoreActions(
        state_dir=args.recovery_state,
        rename_databases_from_to=rename_databases_from_to,
        source_max_connections=args.source_max_connections,
        pg=Posgres(
            pg_bin=args.pg_bin,
            pg_data=args.pg_data,
            pg_config=args.pg_config,
        )
    )

    log.info('Start restore')
    try:
        restore()
    except Exception:
        log.exception('Got unexpected exception')
        raise


if __name__ == '__main__':
    main()
