#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Check if we are master and do backup
"""

import argparse
import logging
import logging.handlers
import subprocess

import psycopg2


def _main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-k', '--keep', type=int, default=7, help='Keep this many backups')
    parser.add_argument(
        '-i', '--io_limit', type=int, default=5242880,
        help='Read io limit (bytes)')
    parser.add_argument(
        '-d',
        '--pgdata',
        type=str,
        default='/var/lib/postgresql/9.6/data',
        help='PGDATA path')
    parser.add_argument('-p', '--pool', type=int, default=4, help='Pool size')
    args = parser.parse_args()

    log = logging.getLogger('main')
    log.setLevel(logging.DEBUG)
    log_handler = logging.handlers.RotatingFileHandler(
        '/var/log/postgresql/s3-backup.log',
        mode='a', maxBytes=10*1024*1024,
        backupCount=4, encoding=None, delay=0)
    log_handler.setFormatter(
        logging.Formatter('%(asctime)s [%(levelname)s]: %(message)s'))
    log.addHandler(log_handler)

    try:
        conn = psycopg2.connect('dbname=postgres')
        cur = conn.cursor()
        cur.execute('show transaction_read_only')

        if cur.fetchone()[0] == 'off':
            log.info('Starting backup')
            subprocess.check_call(
                [
                    '/usr/bin/envdir', '/etc/wal-e/envdir',
                    '/usr/bin/timeout', '86400',
                    '/usr/bin/wal-e', 'backup-push',
                    '--cluster-read-rate-limit=%s' % args.io_limit,
                    '--pool-size=%s' % args.pool, args.pgdata,
                ],
                stdout=log_handler.stream,
                stderr=log_handler.stream,
                cwd='/tmp')
            subprocess.check_call(
                [
                    '/usr/bin/envdir', '/etc/wal-e/envdir', '/usr/bin/wal-e',
                    'delete', '--confirm', 'retain', str(args.keep),
                ],
                stdout=log_handler.stream,
                stderr=log_handler.stream,
                cwd='/tmp')
        else:
            log.info('Replica. Exiting')
    except Exception as exc:
        log.error('Unable to make backup: %s', repr(exc))


if __name__ == '__main__':
    _main()
