#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Check if we are master and do backup
"""

import argparse
import datetime
import logging
import logging.handlers
import os
import socket
import subprocess
import time

import psycopg2
from kazoo.client import KazooClient

try:
    from configparser import ConfigParser
except ImportError:
    from ConfigParser import ConfigParser


def get_config(path):
    """
    Parse config from path
    """
    config = ConfigParser()
    config.read(path)
    return config


def get_logger(config):
    """
    Initialize logger
    """
    log = logging.getLogger('main')
    log.setLevel(logging.DEBUG)
    log_handler = logging.handlers.RotatingFileHandler(
        config.get('main', 'log_path'),
        mode='a',
        maxBytes=10 * 1024 * 1024,
        backupCount=4,
        encoding=None,
        delay=0)
    log_handler.setFormatter(
        logging.Formatter('%(asctime)s [%(levelname)s]: %(message)s'))
    log.addHandler(log_handler)

    return log, log_handler


def get_pg_info(log, config):
    """
    Get status of local postgresql host
    """
    info = {
        'alive': False,
        'replica': False,
    }
    try:
        conn = psycopg2.connect('dbname=postgres')
        cur = conn.cursor()
        cur.execute('SELECT pg_is_in_recovery()')
        info['alive'] = True
        if cur.fetchone()[0]:
            info['replica'] = True
            cur.execute('SELECT extract(epoch FROM '
                        '(current_timestamp - ts)) FROM repl_mon')
            if cur.fetchone()[0] > config.getint('main',
                                                 'max_replication_lag'):
                info['alive'] = False
    except Exception as exc:
        log.error('PostgreSQL is dead: %s', repr(exc))

    return info


def get_pgsync_role(zk_client, hostname, config):
    """
    Get role of hostname using pgsync
    """
    leader_lock = zk_client.Lock(
        os.path.join(config.get('main', 'pgsync_prefix'), 'leader'))
    sync_lock = zk_client.Lock(
        os.path.join(config.get('main', 'pgsync_prefix'), 'sync_replica'))
    leader_contenders = leader_lock.contenders()
    sync_contenders = sync_lock.contenders()
    if leader_contenders and leader_contenders[0] == hostname:
        return 'master'
    elif sync_contenders and sync_contenders[0] == hostname:
        return 'sync'
    return 'async'


def election(log, config, pg_info):
    """
    Participate in zk-based election
    """
    try:
        deadline = time.time() + config.getint('main', 'election_timeout')
        log.info('Starting election with deadline: %s', deadline)
        hostname = socket.getfqdn()
        zk_client = KazooClient(config.get('main', 'zk_hosts'))
        zk_client.start()
        election_lock = zk_client.Lock(
            config.get('main', 'election_lock'), hostname)
        if config.getboolean('main', 'use_pgsync'):
            role = get_pgsync_role(zk_client, hostname, config)
        else:
            role = 'async' if pg_info['replica'] else 'master'
        log.info('Local host role is %s', role)
        while time.time() < deadline:
            contenders = election_lock.contenders()
            if not contenders or (role == 'async' and
                                  hostname not in contenders):
                log.info('Acquiring election lock')
                election_lock.acquire(timeout=10)
            else:
                if len(contenders) > 1 and contenders[0] == hostname:
                    if role in ('master', 'sync'):
                        log.info('We are %s: releasing election lock', role)
                        election_lock.release()
                        contenders = contenders[1:]
            time.sleep(1)
        if contenders[0] == hostname:
            while len(contenders) != 1:
                log.info('Waiting for other nodes to stop election: %s',
                         ', '.join(contenders[1:]))
                time.sleep(1)
            return True
        return False
    except Exception as exc:
        log.error('election error: %s', repr(exc))
        return False


def do_backup(log, pgdata, log_stream, timeout):
    """
    Run wal-g
    """
    log.info('Starting backup')

    subprocess.check_call(
        [
            '/usr/bin/envdir',
            '/etc/wal-g/envdir',
            '/usr/bin/timeout',
            str(timeout),
            '/usr/bin/wal-g',
            'backup-push',
            pgdata,
        ],
        stdout=log_stream,
        stderr=log_stream,
        cwd='/tmp')


def delete_old_backups(log, keep, log_stream):
    """
    Delete old backups up to keep num
    """
    deleted_before = datetime.datetime.utcnow() - datetime.timedelta(days=keep)
    log.info('Delete old backups')

    subprocess.check_call(
        [
            '/usr/bin/envdir',
            '/etc/wal-g/envdir',
            '/usr/bin/timeout',
            '86400',
            '/usr/bin/wal-g',
            'delete',
            'before',
            'FIND_FULL',
            deleted_before.strftime('%Y-%m-%dT%H:%M:%SZ'),
            '--confirm',
        ],
        stdout=log_stream,
        stderr=log_stream,
        cwd='/tmp')


def main():
    """
    Console entry-point
    """
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-c',
        '--config',
        type=str,
        default='/etc/wal-g-backup-push.conf',
        help='config path')
    parser.add_argument(
        '--skip-delete-old',
        action='store_true',
        help="Don't delete old backups",
    )
    args = parser.parse_args()

    config = get_config(args.config)
    log, log_handler = get_logger(config)

    try:
        info = get_pg_info(log, config)
        if not info['alive']:
            log.info('Not participating in election')
            return
        if not election(log, config, info):
            log.info('Election lost. Skipping backup')
            return
        do_backup(log, config.get('main', 'pgdata'),
                  log_handler.stream, config.get('main', 'timeout'))
        if not args.skip_delete_old:
            delete_old_backups(log, config.getint('main', 'keep'),
                               log_handler.stream)
    except Exception as exc:
        log.exception('Unable to make backup')
        raise


if __name__ == '__main__':
    main()
