#!/usr/bin/env python

from collections import Counter
from subprocess import check_output

import requests

ATTEMPTS = 3
YASM_SUFFIX = '_txxv'
YASM_METRIC_PREFIX = 'sharpei_masters_distribution_'


def get_stat():
    for i in range(ATTEMPTS):
        try:
            result = requests.get('http://localhost:9999/stat?force=1')
            result.raise_for_status()
            return result.json()
        except Exception:
            continue
    return None


def count_masters_distribution(lowercase=True):
    distribution = Counter()
    stat = get_stat()
    if not stat:
        return distribution
    for shard_id, props in stat.items():
        for db in props['databases']:
            if db['role'] == 'master':
                dc = db.get('address', {}).get('dataCenter')
                if lowercase:
                    dc = dc.lower()
                distribution[dc] += 1
    return distribution


def generate_yasm_metrics():
    return [
        {
            'name': YASM_METRIC_PREFIX + dc + YASM_SUFFIX,
            'val': cnt
        }
        for dc, cnt in count_masters_distribution().items()
    ]


def get_yasm_tags():
    tags = {}
    cmd = ['/usr/local/yasmagent/mail_postgresql_getter.py']
    for line in check_output(cmd, shell=False, universal_newlines=True).split('\n'):
        line_parts = line.split()
        if not line_parts or not line_parts[0].endswith('mailsharpei'):
            continue
        for line_part in line_parts[1:]:
            if not line_part.startswith('a_'):
                continue
            _, tag_name, tag_value = line_part.split('_', 3)
            tags[tag_name] = tag_value
    return tags


def post_yasm_metrics():
    yasm_params = [{
        "tags": get_yasm_tags(),
        "ttl": 60,
        "values": generate_yasm_metrics(),
    }]
    requests.post('http://localhost:11005', json=yasm_params)


if __name__ == '__main__':
    post_yasm_metrics()
