import argparse
import requests
import os
import re
import time
import urllib3
urllib3.disable_warnings()

import logging
logging.basicConfig(level=logging.INFO, format="[%(asctime)s] %(levelname)s %(name)s %(message)s")
logger = logging.getLogger(__name__)

# https://deploy.yandex-team.ru/docs/reference/tools/dctl
YP_TOKEN = os.environ.get('DCTL_YP_TOKEN')
assert YP_TOKEN

# https://wiki.yandex-team.ru/awacs/api/#avtorizacija
NANNY_TOKEN = os.environ.get('NANNY_TOKEN')
assert NANNY_TOKEN

default_awacs_headers = {
    'Accept': 'application/json, text/plain, */*',
    'Content-Type': 'application/json',
    'Authorization': 'OAuth %s' % NANNY_TOKEN
}
dns_zone = 'in.yandex-team.ru'
banned_dcs = {'man'}


def get_deploy_spec(stage):
    url = 'https://xdc.yp.yandex-team.ru:8443/ObjectService/GetObjects'
    headers = {
        'Accept': 'application/json, text/plain, */*',
        'Content-Type': 'application/json',
        'Authorization': 'OAuth %s' % YP_TOKEN
    }
    data_raw = {"object_type":"stage","selector":{"paths":["/meta","/labels","/spec","/status"]},"subrequests":[{"object_id":stage}],"format":1,"options":{"fetch_timestamps":False}}
    rq = requests.post(url, json=data_raw, headers=headers, verify=False)
    assert rq.status_code == 200
    return rq.json()


# TODO: more parsing scenarios for different configurations
def parse_deploy_spec(spec):
    res = {}
    spec_units = spec['subresponses'][0]['result']['value_payloads'][2]['yson']['deploy_units']
    for key, du in spec_units.items():
        unit = {"network_id": du['network_defaults']['network_id']}
        if du.get('multi_cluster_replica_set'):
            unit.update({'clusters': [c['cluster'] for c in du['multi_cluster_replica_set']['replica_set']['clusters']]})
        elif du.get('replica_set'):
            unit.update({'clusters': [c for c in du['replica_set']['per_cluster_settings'].keys()]})
        res[key] = unit
    return res


def awacs_create_namespace(name, is_testing=False, category='disk'):
    url = 'https://awacs.yandex-team.ru/api/CreateNamespace/'
    headers = default_awacs_headers
    data_raw = {
        "meta": {
            "id": name,
            "category": category, "auth": {"type": "STAFF", "staff": {"owners": {"logins": [], "groupIds": ["380"]}}},
            "webauth": {"responsible": {"logins": []}}, "abcServiceId": 406
        },
        "order": {
            "flowType": "EMPTY", "alertingSimpleSettings": {"notifyStaffGroupId": 28101},
            "envType": {True: 'NS_ENV_TESTING', False: 'NS_ENV_PRODUCTION'}.get(is_testing)
        }
    }
    rq = requests.post(url, json=data_raw, headers=headers, verify=False)
    if rq.status_code == 409:
        logger.info('Namespace %s already exists' % name)
        return
    assert rq.status_code == 200
    logger.info('Creating namespace "%s"' % name)
    wait_for_namespace(name)
    return rq


def awacs_create_backend(namespace_name, stage_name, spec, backend_name=None):
    backend_name = backend_name or stage_name
    url = 'https://awacs.yandex-team.ru/api/CreateBackend/'
    headers = default_awacs_headers

    endpoint_sets = [{'cluster': c, 'endpointSetId': '%s.%s' % (stage_name, du)} for du, conf in spec.items() for c in
                     conf['clusters'] if c not in banned_dcs]
    data_raw = {
        "meta": {"id": backend_name, "namespace_id": namespace_name, "comment": "Initial commit",
                 "auth": {"type": "STAFF", "staff": {"owners": {"logins": [], "group_ids": []}}}}, "spec": {
            "selector": {"type": "YP_ENDPOINT_SETS", "allowEmptyYpEndpointSets": False, "nannySnapshots": [],
                         "gencfgGroups": [],
                         "ypEndpointSets": endpoint_sets,
                         "balancers": []}, "labels": {}}, "validateYpEndpointSets": True}
    rq = requests.post(url, json=data_raw, headers=headers, verify=False)
    if rq.status_code == 409:
        logger.info('Backend %s already exists' % backend_name)
        return
    assert rq.status_code == 200
    time.sleep(5)
    logger.info('Backend "%s" created' % backend_name)
    return rq


def awacs_create_dns(namespace_name, backend, address=None):
    address = address or convert_name(backend)
    url = 'https://awacs.yandex-team.ru/api/CreateDnsRecord/'
    headers = default_awacs_headers
    dns_record = '%s.in.yandex-team.ru' % address
    data_raw = {'meta': {'id': '%s.in.yandex-team.ru' % address, 'namespaceId': namespace_name,
                         'comment': 'Initial commit',
                         'auth': {'type': 'STAFF', 'staff': {'owners': {'logins': [], 'group_ids': []}}}},
                'order': {'type': 'ADDRESS', 'nameServer': {'id': dns_zone, 'namespaceId': 'infra'},
                          'address': {
                              'zone': address, 'balancingPolicy': {'type': 'ROUNDROBIN'},
                              'backends': {'type': 'EXPLICIT', 'backends': [
                                  {'id': backend, 'namespaceId': namespace_name}]}},
                          'ctlVersion': 1}}
    rq = requests.post(url, json=data_raw, headers=headers, verify=False)
    if rq.status_code == 409:
        logger.info('Record %s already exists' % dns_record)
        return
    assert rq.status_code == 200
    logger.info('Creating dns record "%s"' % dns_record)
    wait_for_dns(dns_record, namespace_name)
    return rq


def wait_for_namespace(name):
    url = 'https://awacs.yandex-team.ru/api/GetNamespace/'
    headers = default_awacs_headers
    data_raw = {"id": name, "consistency": "STRONG"}
    while True:
        rq = requests.post(url, json=data_raw, headers=headers, verify=False)
        assert rq.status_code == 200
        status = rq.json()['namespace']['order']['status']['status']
        logger.info('Namespace "%s" status: %s' % (name, status))
        if status == 'FINISHED':
            return
        time.sleep(10)


def wait_for_dns(address, namespace):
    url = 'https://awacs.yandex-team.ru/api/GetDnsRecord/'
    headers = default_awacs_headers
    data_raw = {"id": address, "namespaceId": namespace, "consistency": "STRONG"}
    while True:
        rq = requests.post(url, json=data_raw, headers=headers, verify=False)
        assert rq.status_code == 200
        status = rq.json()['dnsRecord']['order'].get('status', {}).get('status', 'UNKNOWN')
        logger.info('DNS record "%s" status: %s' % (address, status))
        if status == 'FINISHED':
            return
        time.sleep(10)


def convert_name(name):
    pattern = re.compile(r'^[a-z][a-z0-9-.]+$')
    name = re.sub(r'_', '-', name)
    name = re.sub(r'[^a-z0-9-.]', '', name)
    assert pattern.match(name)
    return name


def create_all(stage, is_testing=False,):
    namespace = '%s.%s' % (convert_name(stage), dns_zone)
    spec = parse_deploy_spec(get_deploy_spec(stage))
    awacs_create_namespace(namespace, is_testing)
    awacs_create_backend(namespace, stage, spec)
    awacs_create_dns(namespace, stage)
    logger.info("Done https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/%s/show" % namespace)


def main():
    parser = argparse.ArgumentParser(description='Create awacs admin namespace/backends/dns.')
    parser.add_argument('-s', '--stage', type=str, required=True, help='Deploy stage name')
    parser.add_argument('-t', '--testing', default=False, action='store_true', help='Testing flag')
    args = parser.parse_args()
    create_all(args.stage, args.testing)


if __name__ == "__main__":
    main()


