#!/usr/bin/python
import argparse
import json
import os
import yaml
from pathlib import Path

# def get_qloud_spec(qloud_env):
#     qloud_spec_url = 'https://qloud-ext.yandex-team.ru/api/v1/environment/dump/'
#     url = urljoin(qloud_spec_url, qloud_env)
#     headers = {'authorization': 'OAuth ' + QLOUD_TOKEN}
#     r = requests.get(url, headers=headers)
#     qloud_spec = json.loads(r.text)
#     return qloud_spec

def get_dict_rec_value(data, keys):
    # dict[a,b,c] -> dict[a][b][c]
    if not isinstance(data, dict):
        return data
    return get_dict_rec_value(data[keys[0]], keys[1:]) if keys else data


def update_dict_rec_value(dct, path, val):
    # dict[a,b,c] = d -> dict[a][b][c] = d
    cpath = path.pop(0)
    if not path:
        dct[cpath] = val
        return dct
    else:
        dct[cpath] = update_dict_rec_value(dct[cpath], path, val)
        return dct


def _mergedicts(dict1, dict2):
    for k in set(dict1.keys()).union(dict2.keys()):
        if k in dict1 and k in dict2:
            if isinstance(dict1[k], dict) and isinstance(dict2[k], dict):
                yield (k, dict(_mergedicts(dict1[k], dict2[k])))
            else:
                # If one of the values is not a dict, you can't continue merging it.
                # Value from second dict overrides one in first and we move on.
                yield (k, dict2[k])
                # Alternatively, replace this with exception raiser to alert you of value conflicts
        elif k in dict1:
            yield (k, dict1[k])
        else:
            yield (k, dict2[k])


def merge_dicts(dict1, dict2, exception_paths=[]):
    # Merge dict2 into dict1, except in paths like [['a', 'b', 'c'], ['a', 'd']]
    new_dict = dict(_mergedicts(dict2, dict1))
    for path in exception_paths:
        new_dict = update_dict_rec_value(new_dict, path, get_dict_rec_value(dict2, path))
    return new_dict


def get_spec(file_path):
    with open(file_path) as inpf:
        data = yaml.safe_load(inpf)
        return data

def list_deploy_units(spec):
    return tuple(spec['spec']['deploy_units'].keys())

def get_unit(spec, unit_name):
    return spec['spec']['deploy_units'][unit_name]


def merge_dus(du1, du2, clusters=True, resources=True, readiness=True, secrets=True, envs=False):
    dct_res = du2.copy()

    if du1.get('multi_cluster_replica_set'):
        base_path = ['multi_cluster_replica_set', 'replica_set']
    else:
        base_path = ['replica_set', 'replica_set_template']

    if clusters:
        clusters_paths = base_path + ['clusters']
        dct_res = update_dict_rec_value(dct_res, clusters_paths, get_dict_rec_value(du1, clusters_paths))

    if resources:
        boxes_paths = base_path + ['pod_template_spec', 'spec', 'pod_agent_payload', 'spec', 'boxes']
        dct_res = update_dict_rec_value(dct_res, boxes_paths, [merge_dicts(
            get_dict_rec_value(du1, boxes_paths)[0],
            get_dict_rec_value(du2, boxes_paths)[0], [['id']])])

        resources_paths = base_path + ['pod_template_spec', 'spec', 'pod_agent_payload', 'spec', 'resources']
        dct_res = update_dict_rec_value(dct_res, resources_paths, merge_dicts(
            get_dict_rec_value(du1, resources_paths),
            get_dict_rec_value(du2, resources_paths)))

    if readiness:
        readiness_path = base_path + ['pod_template_spec', 'spec', 'pod_agent_payload', 'spec', 'workloads']
        readiness_dct = update_dict_rec_value(get_dict_rec_value(du2, readiness_path)[0], ['readiness_check'],
                                              get_dict_rec_value(du1, readiness_path)[0]['readiness_check'])
        dct_res = update_dict_rec_value(dct_res, readiness_path, [readiness_dct])

    if envs:
        readiness_path = base_path + ['pod_template_spec', 'spec', 'pod_agent_payload', 'spec', 'workloads']
        envs_dct = update_dict_rec_value(get_dict_rec_value(du2, readiness_path)[0], ['env'],
                                              get_dict_rec_value(du1, readiness_path)[0]['env'])
        dct_res = update_dict_rec_value(dct_res, readiness_path, [envs_dct])

    if secrets:
        secrets_path = base_path + ['pod_template_spec', 'spec', 'secret_refs']
        dct_res = update_dict_rec_value(dct_res, secrets_path, merge_dicts(
            get_dict_rec_value(du1, secrets_path),
            get_dict_rec_value(du2, secrets_path)))

    return dct_res


def process_dus(spec, main_du, **merge_keys):
    units = [u for u in list_deploy_units(spec) if u != main_du]
    main_unit = get_unit(spec, main_du)
    res_spec = spec.copy()
    for u in units:
        du_spec = merge_dus(main_unit, get_unit(spec, u), **merge_keys)
        path = ['spec', 'deploy_units', u]
        res_spec = update_dict_rec_value(res_spec, path, du_spec)
    return res_spec


def process_spec(spec, main_du, output_path, args):
    merge_keys = {'clusters': args.clusters,
                  'resources': args.resources,
                  'readiness': args.readiness,
                  'secrets': args.secrets,
                  'envs': args.envs}
    new_spec = process_dus(spec, main_du, **merge_keys)
    with open(output_path, 'w') as yaml_out:
        yaml.safe_dump(new_spec, yaml_out)
        print('Output file: %s' % output_path)


def main():
    parser = argparse.ArgumentParser(description='Expand single deploy unit config to another units.')
    parser.add_argument('path', type=str, help='Path to spec .yaml file')
    parser.add_argument('-d', '--du', type=str, required=False, help='Main deploy unit')

    parser.add_argument('-o', '--output', type=str, required=False, help='Script output directory')
    parser.add_argument('-s', '--suffix', type=str, required=False, default='copypasted', help='Output file suffix')

    parser.add_argument('--no-clusters', dest='clusters', action='store_false', help='Ignore clusters')
    parser.add_argument('--no-resources', dest='resources', action='store_false', help='Ignore resources')
    parser.add_argument('--no-readiness', dest='readiness', action='store_false', help='Ignore readiness')
    parser.add_argument('--no-secrets', dest='secrets', action='store_false', help='Ignore secrets')
    parser.add_argument('--envs', dest='envs', action='store_true', help='Copy environment variables')

    args = parser.parse_args()

    path = Path(args.path)
    spec = get_spec(path)
    dus = list_deploy_units(spec)
    if not args.du or args.du not in dus:
        print("Need to select one deploy unit to spread settings: --du [%s]" % " ".join(dus))
        exit(1)

    output_dir = Path(args.output or path.parent)
    output_file = output_dir / (path.stem + '_%s' % args.suffix + path.suffix)

    process_spec(spec, args.du, output_file, args)


if __name__ == '__main__':
    main()
