#!/usr/bin/env python3
import os
import sys
import gnupg
import logging
import argparse
import shutil

try:
    import vault_client
except ImportError:
    import library.python.vault_client as vault_client

_EMAIL = 'disk-admin@yandex-team.ru'
_KEY_LEN = 4096
_ALGO = 'RSA'

ARMORED_NAME = 'armored.gpg'

YAV_TAGS = 'disk,salt,pg,stable,gpg'
YAV_STAFF_GROUP = 380
YAV_ROBOT_LOGIN = 'robot-disk-cloud'

logging.basicConfig(stream=sys.stdout, level=logging.DEBUG)
LOG = logging.getLogger()


def _gpg_gen(name, path='.gnupg', email=_EMAIL, gpgbinary='gpg'):
    """
    Generate the actual keys.
    Takes a LOOOOOOOOOOOOOOOOOONG time to complete when entropy is low.
    And it usually is.
    """
    try:
        gpg = gnupg.GPG(gnupghome=path, gpgbinary=gpgbinary)
    except TypeError:
        gpg = gnupg.GPG(homedir=path, binary=gpgbinary)
    result = gpg.gen_key(
        gpg.gen_key_input(
            key_type=_ALGO,
            key_length=_KEY_LEN,
            name_real=name,
            name_email=email,
            expire_date=0
        ),
    )
    # Even if you kill gpg with -9, it will return success.
    # Have to list actual keys to check for success.
    keys = {
        'public': gpg.list_keys(),
        'private': gpg.list_keys(secret=True),
    }
    if len(keys['public']) != 1 and len(keys['private']) != 1:
        LOG.error(
            'gpg call failed: keys=%s, stderr=%s',
            keys,
            getattr(result, 'stderr', ''),
        )
        raise RuntimeError(
            'gpg call failed: %s' % getattr(result, 'stderr', ''))

    ascii_armored_private_keys = gpg.export_keys(name, True)
    armored_key_path = os.path.join(path, ARMORED_NAME)
    with open(armored_key_path, 'w') as key_file:
        key_file.write(ascii_armored_private_keys)


def _upload_secret(path, name, secret_file_name=ARMORED_NAME, tags=YAV_TAGS):
    secret_path = os.path.join(path, secret_file_name)
    secret_name = 'pg_gpg.{}'.format(name)

    with open(secret_path, 'r') as secret_file:
        secret_data = secret_file.read()

    yav = vault_client.instances.Production()
    secret_uuid = yav.create_secret(name=secret_name, tags=tags)
    yav.create_secret_version(secret_uuid,
                              value={secret_file_name: secret_data})
    yav.add_user_role_to_secret(secret_uuid=secret_uuid, role='OWNER',
                                staff_id=YAV_STAFF_GROUP)
    yav.add_user_role_to_secret(secret_uuid=secret_uuid, role='READER',
                                login=YAV_ROBOT_LOGIN)

    return secret_uuid


def main():
    parser = argparse.ArgumentParser(
        description='Generate gpg keys for wal-g and upload them to yav.')
    parser.add_argument('-u', '--upload', action='store_true',
                        help='upload armored key to yav secret')
    parser.add_argument('-c', '--clear', action='store_true',
                        help='clear temp directory with gpg keys')
    parser.add_argument('-g', '--gen_gpg_keys', action='store_true',
                        help='generate gpg keys')
    parser.add_argument('-t', '--tags', type=str, help='set custom yav tags',
                        default=YAV_TAGS)
    parser.add_argument(
        'shard',
        help='shard name (hostname without DC letter and domain part)')
    args = parser.parse_args()
    name = args.shard

    path = '.{}'.format(name)

    if args.gen_gpg_keys:
        try:
            os.makedirs(path)
        except OSError as error:
            LOG.error('Can not create tmp workdir: {}'.format(error))
            sys.exit(1)
        LOG.info('Generating secret keys, please wait...')
        _gpg_gen(name, path=path, gpgbinary=os.environ.get('GPG_PATH', 'gpg1'))

    if args.upload:
        secret_uuid = _upload_secret(path, name, tags=args.tags)
        LOG.info('Secret was uploaded with id: {}'.format(secret_uuid))
        print('#' * 16)
        print('gpg-yav-secrets: {{{{ salt.yav.get(\'{}\') | json }}}}'.format(
            secret_uuid))
        print('#' * 16)

    if args.clear:
        LOG.info('Removing temp gpg workdir: {}'.format(path))
        try:
            shutil.rmtree(path)
        except FileNotFoundError:
            LOG.info('No directory removed')


if __name__ == "__main__":
    main()
