#!/usr/bin/python
import argparse
import json
import os
import requests

from jinja2 import Template
from math import pow
from pprint import pprint
from requests.compat import urljoin

QLOUD_TOKEN = os.environ['QLOUD_TOKEN']
assert QLOUD_TOKEN
REGISTRY_TOKEN = os.environ['DOCKER_TOKEN']
assert REGISTRY_TOKEN


def get_qloud_spec(qloud_env):
    qloud_spec_url = 'https://qloud-ext.yandex-team.ru/api/v1/environment/dump/'
    url = urljoin(qloud_spec_url, qloud_env)
    headers = {'authorization': 'OAuth ' + QLOUD_TOKEN}
    r = requests.get(url, headers=headers)
    qloud_spec = json.loads(r.text)
    return qloud_spec


def get_docker_image_info(image_name):
    registry_url = 'https://registry.yandex.net/v2/'
    url = "{}/{}/tags/list".format(registry_url, image_name)
    headers = {'authorization': 'OAuth ' + REGISTRY_TOKEN}
    r = requests.get(url, headers=headers)
    tags = []
    for tag in json.loads(r.text)['tags']:
        try:
            itag = int(tag)
        except ValueError:
            pass
        else:
            tags.append(itag)
    tags.sort()
    return {'image': image_name, 'tag': tags[-1]}


def get_deploy_names(qloud_env, qloud_spec, component_id=0):
    _, app, app_env = qloud_env.split('.')
    deploy_project = app
    deploy_stage = "{}-{}".format(app, app_env)
    deploy_unit = qloud_spec['components'][component_id]['componentName']
    return {'project': deploy_project,
            'stage': deploy_stage,
            'deploy_unit': deploy_unit,
            'box': deploy_unit,
            'workload': deploy_unit}


def get_env_variables(spec, component_id=0):
    qloud_env_component_variables = spec['components'][component_id]['environmentVariables']
    qloud_env_user_variables = spec['userEnvironmentMap']
    qloud_env = {**qloud_env_user_variables, **qloud_env_component_variables}
    return qloud_env


def get_hw_info(qloud_properties):
    ram, cpu, net, io = qloud_properties['size'].split(';')
    return {'ram': int(int(ram) * pow(1024, 3)),
            'cpu': int(float(cpu) * 1000),
            'net': int((int(net) / 8) * pow(1024, 2))}


def get_resources(spec, component_id=0):
    qloud_sandbox_resources = spec['components'][component_id]['sandboxResources']
    return qloud_sandbox_resources


def get_network(qloud_properties):
    network = qloud_properties['network']
    return '_{}_'.format(network)


def get_component_properties(qloud_spec, qloud_env):
    components_amount = len(qloud_spec['components'])
    res = []
    for component_id in range(components_amount):
        qloud_component_type = qloud_spec['components'][component_id]['componentType']
        if qloud_component_type == 'standard':
            qloud_properties = qloud_spec['components'][component_id]['properties']
            qloud_docker_image = qloud_properties['repository'].split(':')
            qloud_docker_image_name = '/'.join(qloud_docker_image[0].split('/')[1:])

            docker_info = get_docker_image_info(qloud_docker_image_name)
            deploy_names = get_deploy_names(qloud_env, qloud_spec, component_id)
            env_variables = get_env_variables(qloud_spec, component_id)
            hw_quota = get_hw_info(qloud_properties)
            resources = get_resources(qloud_spec, component_id)
            network = get_network(qloud_properties)

            # pprint(qloud_spec)
            # pprint(docker_info)
            # pprint(deploy_names)
            # pprint(env_variables)
            # pprint(hw_quota)
            # pprint(resources)

            res.append({
                "deploy_names": deploy_names,
                "docker_info": docker_info,
                "hw_quota": hw_quota,
                "env_variables": env_variables,
                "resources": resources,
                "network": network,
            })
    return res


def render_template(qloud_components, network_id=None):
    with open('./base_yaml/disk-deploy-template-stable.yaml') as template:
        template = Template(template.read(), trim_blocks=True, lstrip_blocks=True)
        if network_id:
            deploy_spec = template.render(qloud_components=qloud_components, network_id=network_id)
        else:
            deploy_spec = template.render(qloud_components=qloud_components)
        return deploy_spec


def generate_jinja(env, output_dir='./', network_id=None):
    qloud_spec = get_qloud_spec(env)
    qloud_components = get_component_properties(qloud_spec, env)
    deploy_spec = render_template(qloud_components, network_id)
    deploy_names = qloud_components[0]['deploy_names']
    spec_filename = "{}.yaml".format(deploy_names['stage'])

    with open(os.path.join(output_dir, spec_filename), 'w') as f:
        f.write(deploy_spec)
        print('Output result: %s' % f.name)


def main():
    parser = argparse.ArgumentParser(description='Convert qloud environment into deploy spec.')
    parser.add_argument('env', type=str, help='Qloud environment like "disk.disk-mworker.stable-x"')
    parser.add_argument('-o', '--output', type=str, default='./', required=False, help='Script output directory')
    parser.add_argument('-n', '--network_id', type=str, default=None, required=False, help='Override network_id')
    args = parser.parse_args()
    generate_jinja(args.env, args.output, args.network_id)



if __name__ == '__main__':
    main()
