import os
import csv
import sys
import re
import requests
from collections import defaultdict
from collections import Counter


class App:
    URLS = [
        'https://oops.yandex-team.ru/api/ui/query/page?expression=gencfg+ALL_UNSORTED_SEARCH_PORTAL_PERSONAL_AND_INFRASTRUCTURE_SERVICES_PERSONAL_PRODUCTS_YANDEX_DISK&page-size=99999',
        'https://oops.yandex-team.ru/api/ui/query/page?expression=conductor+disk_pg&page-size=99999',
    ]
    BAD_SSD_RE = [
        r'.*SSDSC2KW512G8.*',
        r'.*THNSN8960PCSE,*',
        r'.*M510DC.*',
        r'.*SSDSA2BW600G3.*',
    ]

    DISK_CLASS_FILENAME = os.path.join(os.path.dirname(sys.argv[0]),
                                       'disk_class.csv')

    def __init__(self):
        self.hosts = self.get_hosts()
        self.disks, self.need_update = self.get_disks()
        self.classes = self.read_disk_class()

    def get_hosts(self):
        hosts = []
        seen = set()
        for url in self.URLS:
            data = self._get_info(url)
            for host in data.get('pageCollection', []):
                bot_id = host['botStatus']['id']
                if bot_id not in seen:
                    seen.add(bot_id)
                    hosts.append(host)
        return hosts

    @classmethod
    def read_disk_class(cls, filename=DISK_CLASS_FILENAME):
        class_dict = {}
        with open(filename, 'r') as csvfile:
            reader = csv.DictReader(csvfile)
            for line in reader:
                model = line['np_segment1']
                match = re.search(r'(.*)\(', model)
                if match:
                    model = match.group(1)
                class_dict[model] = {
                    "p_class": line.get('column2'),
                    "e_class": line.get('column3'),
                }
        return class_dict

    @staticmethod
    def _get_token():
        for k, v in os.environ.items():
            if 'TOKEN' in k and 'AQAD' in v:
                return v
        raise ValueError('oAuth Token not found')

    @classmethod
    def _get_info(cls, url):
        headers = {
            'Authorization': 'OAuth %s' % cls._get_token()
        }
        r = requests.get(url, headers=headers)
        r.raise_for_status()
        return r.json()

    def _get_disk_class(self, name):
        for k, v in self.classes.items():
            if k in name:
                return v
            regex = 'r%s' % k
            match = re.search(regex, name)
            if match:
                return v
        return None

    def disk_iter(self):
        for host in self.hosts:
            inventory = host.get('botInventory', {})
            if not inventory:
                continue
            components = inventory.get('components', [])
            for component in components:
                if component.get('type') == 'HDD':
                    model = component.get('model', '')
                    yield host, component, model

    def get_disks(self):
        need_update = defaultdict(list)
        disks = []
        for host, component, model in self.disk_iter():
            disks.append(model)
            for re_ex in self.BAD_SSD_RE:
                if re.match(re_ex, model, re.I):
                    need_update[host['fqdn']].append(component)
        return disks, need_update

    def disk_by_class(self, key):
        disk_by_class = defaultdict(list)
        for disk in self.disk_ssd:
            classes = self._get_disk_class(disk)
            if classes:
                disk_by_class[classes[key]].append(disk)
            else:
                disk_by_class['unknown'].append(disk)
        return disk_by_class

    @property
    def disk_by_class_e(self):
        return self.disk_by_class('e_class')

    @property
    def disk_by_class_p(self):
        return self.disk_by_class('p_class')

    @property
    def disk_ssd(self):
        return list(filter(lambda x: 'SSD' in x, self.disks))

    def find_hosts_with_model(self, search):
        hosts = []
        for host, component, model in self.disk_iter():
            if search.lower() in model.lower():
                hosts.append(
                    (host['botStatus']['fqdn'], host['botStatus']['id']))
        return hosts


def main():
    app = App()
    print("Total disks: %d" % len(app.disks))
    print("Total SSD: %d" % len(app.disk_ssd))
    print("Disks by E class:")
    for c, disks in app.disk_by_class_e.items():
        counter = Counter(disks)
        print("Class %s (total: %d)" % (c, sum(counter.values())))
        for model, count in counter.items():
            print("%s %d" % (model, count))

    print("Disks by P class:")
    for c, disks in app.disk_by_class_p.items():
        counter = Counter(disks)
        print("Class %s (total: %d)" % (c, sum(counter.values())))
        for model, count in counter.items():
            print("%s %d" % (model, count))


if __name__ == '__main__':
    main()
