#!/usr/bin/env python3

import os
import sys
from kazoo.client import KazooClient
import requests
import json
import subprocess
import time
import logging
import hashlib

logger = logging.getLogger(__name__)

ENV = {
    'testing': {
        'zk_hosts': "zk1e.dst.yandex.net:2181,zk1f.dst.yandex.net:2181,zk1h.dst.yandex.net:2181",
        'endpoints': ['telemost-testing.xmpp%d' % x for x in (1, 2, 3)],
        'zk_prefix': '/telemost/backend',
        'stage_name': 'telemost-testing',
    },
    'prestable': {
        'zk_hosts': "zk1e.dst.yandex.net:2181,zk1f.dst.yandex.net:2181,zk1h.dst.yandex.net:2181",
        'endpoints': ['telemost-testing.prestable%d' % x for x in (1, 2)],
        'zk_prefix': '/telemost/backend-prestable',
        'stage_name': 'telemost-testing',
    },
    'production': {
        'zk_hosts': 'zk01e.disk.yandex.net:2181,zk02e.disk.yandex.net:2181,zk01f.disk.yandex.net:2181,zk02f.disk.yandex.net:2181,zk01h.disk.yandex.net:2181,zk02h.disk.yandex.net:2181,zk02v.disk.yandex.net:2181,zk01v.disk.yandex.net:2181',
        'endpoints': ['telemost-xmpp-prod.xmpp%d' % x for x in
                      (1, 2, 3, 4, 5, 6)],
        'zk_prefix': '/telemost/backend',
        'stage_name': 'telemost-xmpp-prod',
    }
}


class ServiceDiscoveryError(Exception):
    pass


class ServiceDiscovery:
    BASE_URL = 'http://sd.yandex.net:8080'
    CLIENT_NAME = 'telemost-sd'

    def __init__(self, endpoint, dc, only_ready=False):
        self.endpoint = endpoint
        self.dc = dc
        self.only_ready = only_ready

    def get_data(self):
        data = {
            'cluster_name': self.dc,
            'endpoint_set_id': self.endpoint,
            'client_name': self.CLIENT_NAME,
        }
        return data

    def request(self):
        url = self.BASE_URL + '/resolve_endpoints/json'
        data = self.get_data()
        resp = requests.get(url, data=json.dumps(data), timeout=2)
        resp.raise_for_status()
        return resp.json()

    def resolve(self):
        sd = self.request()
        services = []
        endpoint_set = sd.get('endpoint_set')
        if not endpoint_set:
            raise ServiceDiscoveryError(sd)
        if endpoint_set['endpoint_set_id'] != self.endpoint:
            raise ServiceDiscoveryError(sd)
        for endpoint in endpoint_set['endpoints']:
            if self.only_ready and not endpoint['ready']:
                continue
            endpoint['endpoint'] = self.endpoint
            endpoint['dc'] = self.dc
            services.append(endpoint)
        return services


class App:
    def __init__(self, zk_hosts, zk_prefix, endpoints, stage_name,
                 bucket_size=20,
                 dry_run=False,
                 dcs=('sas', 'man', 'vla')):
        self.zk_hosts = zk_hosts
        self.zk_prefix = zk_prefix
        self.endpoints = endpoints
        self.dcs = dcs
        self.backends = self.get_backends()
        self.stage_name = stage_name
        self.zk = KazooClient(hosts=self.zk_hosts)
        self.zk.start()
        self.bucket_size = bucket_size
        self.dry_run = dry_run

    def get_backends(self):
        """
        get backends from YP
        :return:
        """
        backends = []
        for dc in self.dcs:
            for endpoint in self.endpoints:
                sd = ServiceDiscovery(endpoint, dc)
                try:
                    backends.extend(sd.resolve())
                except ServiceDiscoveryError:
                    pass
        logger.debug("Backends from YP: %s" % backends)
        return backends

    def zk_init(self):
        """BE AWARE!!! REINIT ALL BACKENDS"""
        self.zk.ensure_path(self.zk_prefix)
        for x in range(self.bucket_size):
            path = '%s/%s' % (self.zk_prefix, x)
            self.zk.ensure_path(path)
            backend = self.backends[x % (len(self.backends))]
            print(backend)
            print(self.zk.set(path, backend['fqdn'].encode('utf-8')))

    def backends_in_zk_long(self):
        # TODO: replace
        res = []
        try:
            for x in range(self.bucket_size):
                path = "%s/%s" % (self.zk_prefix, x)
                r, stat = self.zk.retry(self.zk.get, path)
                res.append([x, r, path])
        except:
            pass
        return res

    @property
    def backends_in_zk(self):
        backends = set()
        for x in range(self.bucket_size):
            path = "%s/%s" % (self.zk_prefix, x)
            r, stat = self.zk.retry(self.zk.get, path)
            backends.add(r.decode('utf-8'))
        return backends

    def get_unit_for_update(self):
        # unit with backend not in zk
        backends = self.get_backends()
        for backend in backends:
            if backend['fqdn'] not in self.backends_in_zk:
                return backend['endpoint']
        raise Exception('Backup unit not found')

    @property
    def _yatool_cmd(self):
        return os.environ.get('YA_TOOL_PATH', 'ya')

    def dctl_put_stage(self, spec_file):
        logger.info("Start deploy")
        if self.dry_run:
            logger.warning("DRU RUN dctl will run here")
        else:
            subprocess.check_call([
                self._yatool_cmd,
                "tool", "dctl", "put", "stage", spec_file
            ])

    def zk_switch_backend(self, src, dst):
        logger.info("ZK switch: %s -> %s" % (src, dst))
        for x in range(self.bucket_size):
            path = "%s/%s" % (self.zk_prefix, x)
            r, stat = self.zk.retry(self.zk.get, path)
            if r.decode('utf-8') == src:
                if not self.dry_run:
                    self.zk.retry(self.zk.set, path, dst.encode('utf-8'))

    def zk_switch_backend_bucket(self, bucket, dst):
        if bucket > self.bucket_size - 1:
            raise Exception('unknown bucket')
        path = "%s/%s" % (self.zk_prefix, bucket)
        self.zk.retry(self.zk.set, path, dst.encode('utf-8'))
        logger.info("ZK switched bucket %s to %s" % (bucket, dst))

    def get_bucket_by_room(self, room):
        bucket = int(hashlib.sha256(room.encode('utf-8')).hexdigest(),
                   16) % self.bucket_size
        backend = None
        for x in self.backends_in_zk_long():
            if x[0] == bucket:
                backend = x[1]
                break
        return bucket, backend

    def zk_switch_dc(self, dc_name):
        backends = [x['fqdn'] for x in
                    filter(lambda x: x['dc'] != dc_name, self.backends)]
        to_switch_backends = [x['fqdn'] for x in
                              filter(lambda x: x['dc'] == dc_name,
                                     self.backends)]
        i = 0
        for zk_backend in self.backends_in_zk:
            if zk_backend in to_switch_backends:
                replace_backend = backends[i % len(backends)]
                logger.warning(
                    "switch %s -> %s" % (zk_backend, replace_backend))
                self.zk_switch_backend(zk_backend, replace_backend)
                i += 1

    def zk_switch_unit(self, src, dst):
        backends = self._get_backend_for_unit(dst)
        to_switch_backends = self._get_backend_for_unit(src)
        i = 0
        for zk_backend in self.backends_in_zk:
            if zk_backend in to_switch_backends:
                replace_backend = backends[i % len(backends)]
                logger.warning(
                    "switch %s -> %s" % (zk_backend, replace_backend))
                self.zk_switch_backend(zk_backend, replace_backend)
                i += 1

    def _get_backend_for_unit(self, unit):
        endpoint = '%s.%s' % (self.stage_name, unit)
        return [x['fqdn'] for x in
                filter(lambda x: x['endpoint'] == endpoint, self.backends)]

    def check_alive(self, backend):
        url = 'http://%s/ping' % backend
        try:
            r = requests.get(url, timeout=0.5)
            r.raise_for_status()
        except:
            return False
        return True

    def check(self):
        dead = set()
        alive = set()
        for backend in self.backends_in_zk:
            if self.check_alive(backend):
                alive.add(backend)
            else:
                dead.add(backend)
        return alive, dead

    def deploy(self, unit_name, spec_file):
        endpoint_name = "%s.%s" % (self.stage_name, unit_name)
        backend_to_update = None
        for backend in self.backends:
            if backend['endpoint'] == endpoint_name:
                backend_to_update = backend
                break
        if not backend_to_update:
            raise Exception('Backend to update not found')
        if backend_to_update['fqdn'] in self.backends_in_zk:
            raise Exception('Deploy unit in use in zk')
        self.dctl_put_stage(spec_file)
        for x in range(60):
            ready = False
            for dc in self.dcs:
                try:
                    sd = ServiceDiscovery(
                        endpoint_name, dc,
                        only_ready=True)
                    ready = sd.resolve()
                except ServiceDiscoveryError:
                    continue
                if ready:
                    break
            if ready:
                last_updated = ready[0]['fqdn']
                logger.info('%s ready' % last_updated)
                break
            logger.info("%s not ready" % unit_name)
            time.sleep(5)


def main():
    logging.basicConfig(
        stream=sys.stdout, level=logging.INFO,
        format='%(asctime)s [%(levelname)s] [%(module)s:%(funcName)s] %(message)s'
    )
    logging.getLogger(__name__).setLevel(logging.DEBUG)

    cmd = sys.argv[1]
    env = ENV[sys.argv[2]]
    app = App(
        env['zk_hosts'], env['zk_prefix'], env['endpoints'], env['stage_name'],
    )
    if cmd == 'deploy':
        app.deploy(sys.argv[3], sys.argv[4])
    elif cmd == 'yp_backends':
        print("\n".join(["%s [%s]" % (x['fqdn'], x['endpoint']) for x in
                         app.get_backends()]))
    elif cmd == 'zk_backends':
        print("\n".join(app.backends_in_zk))
    elif cmd == 'unit_for_update':
        print(app.get_unit_for_update())
    elif cmd == 'zk_switch':
        app.zk_switch_backend(sys.argv[3], sys.argv[4])
    elif cmd == 'zk_switch_bucket':
        app.zk_switch_backend_bucket(int(sys.argv[3]), sys.argv[4])
    elif cmd == 'zk_switch_dc':
        app.zk_switch_dc(sys.argv[3])
    elif cmd == 'zk_switch_unit':
        app.zk_switch_unit(sys.argv[3], sys.argv[4])
    elif cmd == 'zk_buckets':
        print("\n".join(["bucket: %s; backend: %s" % (x[0], x[1]) for x in
                         app.backends_in_zk_long()]))
    elif cmd == 'get_bucket':
        print(app.get_bucket_by_room(sys.argv[3]))
    elif cmd == 'check_backends':
        print("alive: %s; dead: %s" % app.check())
    elif cmd == 'zk_init':
        app.zk_init()
    else:
        raise Exception('Command not found')


if __name__ == '__main__':
    main()
