package ru.yandex.chemodan.app.persapi;

import java.lang.reflect.Field;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;

import com.google.common.util.concurrent.ListenableFuture;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.persapi.acl.AccessType;
import ru.yandex.chemodan.app.persapi.acl.FactAclManager;
import ru.yandex.chemodan.app.persapi.api.ServiceUnavailableException;
import ru.yandex.chemodan.app.persapi.fact.Fact;
import ru.yandex.chemodan.app.persapi.log.FactLogUtils;
import ru.yandex.chemodan.app.persapi.schema.FactSchemaManager;
import ru.yandex.chemodan.app.persapi.util.FactStoreFuture;
import ru.yandex.inside.logbroker.push.PushClient;
import ru.yandex.inside.logbroker.push.results.DataSendResult;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Instrument;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.InstrumentedData;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.monica.util.MapWithRegistrarAndUnregistrar;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author tolmalev
 */
public class FactManager extends DelayingWorkerServiceBeanSupport implements MonicaContainer{
    private static final Logger logger = LoggerFactory.getLogger(FactManager.class);

    private final FactAclManager aclManager;
    private final FactSchemaManager schemaManager;
    private final PushClient pushClient;

    private final AtomicLong sentFactsCount = new AtomicLong(0);
    private final AtomicLong storedFactsCount = new AtomicLong(0);
    private final AtomicLong failedFactsCount = new AtomicLong(0);

    @MonicaMetric
    private final MeterMap sentFacts = new MeterMap();
    @MonicaMetric
    private final InstrumentMap stores = new InstrumentMap();

    @MonicaMetric
    private final MeterMap storedBytes = new MeterMap();

    private final Duration pushTimeout;

    public FactManager(
            FactAclManager factAclManager, FactSchemaManager schemaManager,
            PushClient pushClient, Duration pushTimeout)
    {
        this.aclManager = factAclManager;
        this.schemaManager = schemaManager;
        this.pushClient = pushClient;
        this.pushTimeout = pushTimeout;

        setDelay(Duration.standardMinutes(1));
    }

    public void storeFact(String clientId, Fact fact) {
        FactStoreFuture future = storeFactAsync(clientId, fact);
        waitFactStored(future);
    }

    public void waitFactStored(FactStoreFuture future) {
        Fact fact = future.getFact();
        stores.measure(() -> {
            try {
                DataSendResult result = future.get(pushTimeout.getMillis(), TimeUnit.MILLISECONDS);
                incAndLog(storedFactsCount, "Stored total facts: {}");
                storedBytes.inc(result.bytes, new MetricName(fact.type), UpdateMode.RECURSIVE);
            } catch (TimeoutException | InterruptedException | ExecutionException e) {
                incAndLog(failedFactsCount, "Failed total facts: {}");
                throw new RuntimeException("Failed push to LogBroker", e);
            }
        }, new MetricName(fact.type), UpdateMode.RECURSIVE);
    }

    private void incAndLog(AtomicLong atomicLong, String s) {
        long currentValue = atomicLong.incrementAndGet();
        if (currentValue % 100 == 0) {
            logger.info(s, currentValue);
        }
    }

    public FactStoreFuture storeFactAsync(String clientId, Fact fact) {
        checkStoreAccess(clientId, fact);
        checkFactData(fact);

        if (!pushClient.isAlive()) {
            throw new ServiceUnavailableException("No alive push sessions");
        }
        ListenableFuture<DataSendResult> result = pushClient.sendData(FactLogUtils.getFactLogString(fact).getBytes());

        sentFacts.inc(new MetricName(fact.type), UpdateMode.RECURSIVE);
        incAndLog(sentFactsCount, "Sent total facts: {}");
        return new FactStoreFuture(result, fact);
    }

    public void checkStoreAccess(String clientId, Fact fact) {
        aclManager.checkAccess(clientId, fact.source, fact.type, AccessType.WRITE);
    }

    public void checkFactData(Fact fact) {
        schemaManager.checkSchema(fact.type, fact.data);
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName("fact-manager", new MetricName("facts", "manager"), "FactManager");
    }

    @Override
    protected void execute() throws Exception {
        Field field = stores.getClass().getDeclaredField("map");
        field.setAccessible(true);
        MapWithRegistrarAndUnregistrar<Instrument, InstrumentedData> map =
                (MapWithRegistrarAndUnregistrar<Instrument, InstrumentedData>) field.get(stores);

        ListF<MetricName> metricNames =
                Cf.list(new MetricName()).plus(schemaManager.listKnownTypes().map(MetricName::new));

        metricNames.forEach(name -> {
            storedBytes.get(name);
            sentFacts.get(name);
            map.getOrCreate(Instrument::new, name);
        });
    }
}
