package ru.yandex.chemodan.app.persapi.acl;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.util.ping.PingerChecker;
import ru.yandex.inside.bunker.BunkerClient;
import ru.yandex.inside.bunker.BunkerNodeInfo;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author tolmalev
 */
public abstract class BunkerDynamicRegistryBase<T> extends DelayingWorkerServiceBeanSupport implements PingerChecker {
    private static final Logger logger = LoggerFactory.getLogger(BunkerDynamicRegistryBase.class);

    protected final BunkerClient bunkerClient;
    protected final String baseNode;

    protected MapF<String, T> configByNode = Cf.map();
    protected boolean initialized = false;

    private final ListF<Runnable> updateListeners = Cf.arrayList();

    public BunkerDynamicRegistryBase(BunkerClient bunkerClient, Duration updatePeriod, String baseNode) {
        this.bunkerClient = bunkerClient;
        this.baseNode = baseNode;

        setSleepBeforeFirstRun(false);
        setSleepMode("period");
        setDelay(updatePeriod);
    }

    protected abstract T parseNode(byte[] bytes);

    @Override
    protected void execute() throws Exception {
        MapF<String, T> newConfig = Cf.hashMap();
        ListF<BunkerNodeInfo> nodes;
        try {
            nodes = bunkerClient.list(baseNode);
        } catch (RuntimeException e) {
            logger.error("Failed getting nodes from bunker: {}", e);
            throw e;
        }
        for (BunkerNodeInfo clientNode : nodes) {
            try {
                newConfig.put(clientNode.name, parseNode(bunkerClient.get(clientNode)));
            } catch (Exception e) {
                logger.error("Invalid config for node {}: {}", clientNode.name, e);
                throw e;
            }
        }
        configByNode = newConfig;
        logger.info("Loaded config for nodes: {}", newConfig.keys());
        initialized = true;

        synchronized (updateListeners) {
            updateListeners.forEach(r -> {
                try {
                    r.run();
                } catch (Throwable e) {
                    ExceptionUtils.throwIfUnrecoverable(e);
                    logger.error("Failed to call listener: {}", e);
                }
            });
        }
    }

    public void addUpdateListener(Runnable r) {
        synchronized (updateListeners) {
            updateListeners.add(r);
        }
    }

    public boolean isInitialized() {
        return initialized;
    }

    @Override
    public boolean isActive() {
        return isInitialized();
    }

    public void checkInitialized() {
        Check.isTrue(initialized, "Registry not initialized from bunker");
    }
}
