package ru.yandex.chemodan.app.persapi.acl;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.util.exception.AccessForbiddenException;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author tolmalev
 */
public class FactAclManager implements MonicaContainer {
    private static final Logger logger = LoggerFactory.getLogger(FactAclManager.class);

    private final FactAclRegistry factAclRegistry;

    @MonicaMetric
    private final InstrumentMap checks = new InstrumentMap();

    public FactAclManager(FactAclRegistry factAclRegistry) {
        this.factAclRegistry = factAclRegistry;
    }

    public boolean isFullAccessAllowed(String clientId, String factType, AccessType accessType) {
        AccessLevel fullTypeAccessLevel = factAclRegistry.getFullTypeAccessLevel(clientId, factType);
        logger.trace("Full type access level: {}", fullTypeAccessLevel);
        return fullTypeAccessLevel.isAllowed(accessType);
    }

    public boolean isAccessAllowed(String clientId, String source, String factType, AccessType accessType) {
        if (isFullAccessAllowed(clientId, factType, accessType)) {
            return true;
        }

        AccessLevel typeAndSourceAccessLevel = factAclRegistry.getTypeAndSourceAccessLevel(clientId, source, factType);
        logger.trace("Type-and-source access level: {}", typeAndSourceAccessLevel);

        return typeAndSourceAccessLevel.isAllowed(accessType);
    }

    public ListF<String> getAccessibleSources(String clientId, String type, AccessType accessType) {
        return factAclRegistry.getAccessibleSources(clientId, type, accessType);
    }

    public ListF<String> getTypeHash(String clientId, String type) {
        return factAclRegistry.getTypeHash(clientId, type);
    }

    public void checkAccess(String clientId, String source, String factType, AccessType accessType) {
        checks.measure(() -> {
            logger.trace("Check rights clientId={}, source={}, fact-type={}, access=type={}",
                    clientId, source, factType, accessType);

            if (!isAccessAllowed(clientId, source, factType, accessType)) {
                throw new AccessDeniedException(clientId, source, factType, accessType);

            }
        }, new MetricName(clientId, source, factType), UpdateMode.RECURSIVE);
    }

    public void checkTvmAccess(String clientId, Option<Integer> tvmClientId) {
        ListF<Integer> clientIds = factAclRegistry.getTvmClientIds(clientId);
        if (clientIds.isEmpty()) {
            logger.warn("TVM not configured for for {}. Allow.", clientId);
            return;
        }

        if(!clientIds.containsTs(tvmClientId.getOrElse(-1))) {
            String message = StringUtils
                    .format("TVM access denied for client {}. Allowed ids: {}. Given: {}", clientId, clientIds,
                            tvmClientId);
            logger.error(message);
            throw new AccessForbiddenException(message);
        }
    }

    public boolean isInitialized() {
        return factAclRegistry.isInitialized();
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName("fact-manager", new MetricName("facts", "acl-manager"), "FactAclManager");
    }
}
