package ru.yandex.chemodan.app.persapi.acl;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.bunker.BunkerClient;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.PartPosition;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.annotation.BendingMethod;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.UnmarshallerContext;
import ru.yandex.misc.bender.parse.XmlOrJsonNode;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class FactAclRegistry extends BunkerDynamicRegistryBase<FactAclRegistry.OneClientConfig> {
    private static final Logger logger = LoggerFactory.getLogger(FactAclRegistry.class);

    public FactAclRegistry(BunkerClient bunkerClient, Duration updatePeriod) {
        super(bunkerClient, updatePeriod, "acl");
    }

    public ListF<Integer> getTvmClientIds(String clientId) {
        return configByNode
                .getOrElse(clientId, OneClientConfig.EMPTY)
                .tvmClientIds;
    }

    public ListF<Integer> getAllTvmClientIds() {
        return configByNode.values().flatMap(c -> c.tvmClientIds);
    }

    public AccessLevel getFullTypeAccessLevel(String clientId, String factType) {
        return configByNode
                .getOrElse(clientId, OneClientConfig.EMPTY)
                .byType
                .getOrElse(factType, AccessLevel.DENY_ALL);
    }

    public AccessLevel getTypeAndSourceAccessLevel(String clientId, String source, String factType) {
        return configByNode
                .getOrElse(clientId, OneClientConfig.EMPTY)
                .bySource
                .getOrElse(source, Cf.map())
                .getOrElse(factType, AccessLevel.DENY_ALL);
    }

    public ListF<String> getAccessibleSources(String clientId, String factType, AccessType accessType) {
        return configByNode
                .getOrElse(clientId, OneClientConfig.EMPTY)
                .bySource
                .filterValues(v -> v.getOrElse(factType, AccessLevel.DENY_ALL).isAllowed(accessType))
                .keys();
    }

    public ListF<String> getTypeHash(String clientId, String factType) {
        return configByNode
                .getOrElse(clientId, OneClientConfig.EMPTY)
                .hashByType
                .getOrElse(factType, Cf.list());
    }

    @Override
    public OneClientConfig parseNode(byte[] bytes) {
        OneClientConfigRaw rawConfig = Bender.jsonParser(OneClientConfigRaw.class).parseJson(bytes);

        MapF<String, AccessLevel> byType = Cf.hashMap();
        MapF<String, MapF<String, AccessLevel>> bySource = Cf.hashMap();
        MapF<String, ListF<String>> hashByType = rawConfig.hashByType.toMap(t -> t.type, t -> t.hash);

        updateConfig(rawConfig.readOnly, byType, bySource, AccessLevel.READ_ONLY);
        updateConfig(rawConfig.readWrite, byType, bySource, AccessLevel.READ_WRITE);
        updateConfig(rawConfig.writeOnly, byType, bySource, AccessLevel.WRITE_ONLY);

        return new OneClientConfig(byType, bySource, hashByType, rawConfig.tvmClientIds);
    }

    private static void updateConfig(ListF<TypeWithMaybeSource> list,
                              MapF<String, AccessLevel> byType,
                              MapF<String, MapF<String, AccessLevel>> bySource,
                              AccessLevel accessLevel)
    {
        for (TypeWithMaybeSource element : list) {
            if (element.source.isPresent()) {
                bySource
                        .getOrElseUpdate(element.source.get(), Cf.hashMap())
                        .put(element.type, accessLevel);
            } else {
                byType.put(element.type, accessLevel);
            }
        }
    }

    @BenderBindAllFields
    public static final class OneClientConfig extends DefaultObject {
        static final OneClientConfig EMPTY = new OneClientConfig(Cf.map(), Cf.map(), Cf.map(), Cf.list());

        private final MapF<String, AccessLevel> byType;
        private final MapF<String, MapF<String, AccessLevel>> bySource;
        private final MapF<String, ListF<String>> hashByType;
        private final ListF<Integer> tvmClientIds;

        public OneClientConfig(
                MapF<String, AccessLevel> byType,
                MapF<String, MapF<String, AccessLevel>> bySource,
                MapF<String, ListF<String>> hashByType,
                ListF<Integer> tvmClientIds)
        {
            this.byType = byType;
            this.bySource = bySource;
            this.hashByType = hashByType;
            this.tvmClientIds = tvmClientIds;
        }
    }

    @BenderBindAllFields
    private static final class OneClientConfigRaw extends DefaultObject {
        @BenderPart(strictName = true, name = "read_only")
        private final ListF<TypeWithMaybeSource> readOnly;

        @BenderPart(strictName = true, name = "read_write")
        private final ListF<TypeWithMaybeSource> readWrite;

        @BenderPart(strictName = true, name = "write_only")
        private final ListF<TypeWithMaybeSource> writeOnly;

        @BenderPart(strictName = true, name = "hash_by_type")
        private final ListF<TypeWithHash> hashByType;

        @BenderPart(strictName = true, name = "tvm_client_ids")
        private final ListF<Integer> tvmClientIds;

        public OneClientConfigRaw(
                ListF<TypeWithMaybeSource> readOnly,
                ListF<TypeWithMaybeSource> readWrite,
                ListF<TypeWithMaybeSource> writeOnly,
                ListF<TypeWithHash> hashByType,
                ListF<Integer> tvmClientIds)
        {
            this.readOnly = readOnly;
            this.readWrite = readWrite;
            this.writeOnly = writeOnly;
            this.hashByType = hashByType;
            this.tvmClientIds = tvmClientIds;
        }
    }

    @BenderBindAllFields
    public static final class TypeWithMaybeSource {
        public final String type;
        public final Option<String> source;

        public TypeWithMaybeSource(String type, Option<String> source) {
            this.type = type;
            this.source = source;
        }

        @BendingMethod
        public static TypeWithMaybeSource parse(XmlOrJsonNode node, PartPosition pp, UnmarshallerContext context) {
            BenderJsonNode json = node.getJson();

            if (json.isValueNode()) {
                return new TypeWithMaybeSource(json.getValueAsString(), Option.empty());
            } else {
                return new TypeWithMaybeSource(
                        json.getField("fact_type").get().getValueAsString(),
                        json.getField("source").map(BenderJsonNode::getValueAsString)
                );
            }
        }
    }

    @BenderBindAllFields
    public static final class TypeWithHash {
        @BenderPart(strictName = true, name = "fact_type")
        public final String type;
        public final ListF<String> hash;

        public TypeWithHash(String type, ListF<String> hash) {
            this.type = type;
            this.hash = hash;
        }
    }

    static FactAclRegistry createForTest(MapF<String, InputStreamSource> configs) {
        FactAclRegistry registry = new FactAclRegistry(null, null);
        registry.stop();
        registry.configByNode = configs.mapValues(iss -> registry.parseNode(iss.readBytes()));
        return registry;
    }
}
