package ru.yandex.chemodan.app.persapi.api;

import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.RetryPolicy;
import org.apache.http.client.HttpClient;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.chemodan.app.persapi.api.quick.YqlQuickFactsRetriever;
import ru.yandex.chemodan.app.persapi.api.yt.RetrieveFactsRunner;
import ru.yandex.chemodan.util.yql.YqlClient;
import ru.yandex.chemodan.util.yt.YtTableNameResolver;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

/**
 * @author yashunsky
 */
@Configuration
public class FactApiYtContextConfiguration {
    @Value("${factapi.axis.yt.logs-root-path}")
    private String ytAxisLogsRootPath;

    @Value("${factapi.axis.yt.root-path}")
    private String ytAxisRootPath;

    @Bean
    public Yt factApiYtClient(
            @Value("${factapi.yt.http.proxy}")
            String ytHttpProxyUrl,
            @Value("${factapi.axis.user.yt.token}")
            String ytUserOauthToken)
    {
        return YtUtils.http(ytHttpProxyUrl, ytUserOauthToken);
    }

    @Bean
    public RetryPolicy factApiYtOperationRetryPolicy(
            @Value("${factapi.yt.operation.retry.count}")
            int ytOperationRetryCount,
            @Value("${factapi.yt.operation.retry.delay}")
            Duration ytOperationRetryDelay)
    {
        return new RetryPolicy()
                .withMaxRetries(ytOperationRetryCount)
                .withDelay(ytOperationRetryDelay.getMillis(), TimeUnit.MILLISECONDS);
    }

    @Bean
    public RetrieveFactsRunner retrieveFactsRunner(
            Yt yt,
            @Qualifier("factApiYtOperationRetryPolicy")
            RetryPolicy retryPolicy)
    {
        return new RetrieveFactsRunner(yt, retryPolicy, YPath.simple(ytAxisRootPath), YPath.simple(ytAxisLogsRootPath));
    }

    @Bean
    public YqlClient yqlClient(
            @Value("${factapi.user-agent}")
            String userAgent,
            @Value("${factapi.yql.uri}")
            String uri,
            @Value("${factapi.axis.user.yql.token}")
            String token,
            @Value("${factapi.yql.http.max-connections}")
            int maxConnections,
            @Value("${factapi.yql.http.timeout}")
            Duration timeout,
            @Value("${factapi.yql.http.retries}")
            int retries)
    {
        HttpClient httpClient = ApacheHttpClientUtils.Builder.create()
                .withHttpsSupport()
                .multiThreaded()
                .withMaxConnections(maxConnections)
                .withTimeout(Timeout.valueOf(timeout))
                .withUserAgent(userAgent)
                .withRequestRetry(retries, true)
                .build();

        return new YqlClient(uri, token, httpClient);
    }

    @Bean
    public YtTableNameResolver ytTableNameResolver(
            Yt yt,
            @Qualifier("factApiYtOperationRetryPolicy")
            RetryPolicy retryPolicy)
    {
        return new YtTableNameResolver(yt, retryPolicy);
    }

    @Bean
    public YqlQuickFactsRetriever yqlQuickFactsRetriever(YqlClient yqlClient, YtTableNameResolver ytTableNameResolver) {
        return new YqlQuickFactsRetriever(yqlClient, ytTableNameResolver);
    }
}
