package ru.yandex.chemodan.app.persapi.api.quick;

import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.IOUtils;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.util.yql.YqlClient;
import ru.yandex.chemodan.util.yql.YqlResponse;
import ru.yandex.chemodan.util.yt.YtTableNameResolver;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
public class YqlQuickFactsRetriever {

    private static final String PY_FACTS_PRECHARGER_NAME = "__main__.py";
    private static final String PY_PRECHARG_FACTS_MR_NAME = "precharge_facts_mr.py";

    private final YqlClient yql;
    private final YtTableNameResolver tableNameResolver;
    private final String proxy;
    private final ListF<String> valid_types;
    private final ListF<String> fields;
    private final YPath ytPrechargedPath;
    private final YPath ytLogsPath;

    public YqlQuickFactsRetriever(YqlClient yql, YtTableNameResolver tableNameResolver) {
        this.yql = yql;
        this.tableNameResolver = tableNameResolver;

        String prechargerScript = getPythonScript(PY_FACTS_PRECHARGER_NAME);

        Matcher matcher = Pattern.compile("PROXY = '(.*?)'\n").matcher(prechargerScript);
        Assert.isTrue(matcher.find(), "Check PROXY definition in " + PY_FACTS_PRECHARGER_NAME);
        proxy = matcher.group(1);

        matcher = Pattern.compile("LOGS_PATH = '(.*?)'\n").matcher(prechargerScript);
        Assert.isTrue(matcher.find(), "Check LOGS_PATH definition in " + PY_FACTS_PRECHARGER_NAME);
        ytLogsPath = YPath.simple(matcher.group(1));

        matcher = Pattern.compile("PRECHARGED_PATH = '(.*?)'\n").matcher(prechargerScript);
        Assert.isTrue(matcher.find(), "Check PRECHARGED_PATH definition in " + PY_FACTS_PRECHARGER_NAME);
        ytPrechargedPath = YPath.simple(matcher.group(1));

        matcher = Pattern.compile("SCHEMA = yson\\.YsonList\\((.*?)\\)", Pattern.DOTALL).matcher(prechargerScript);
        Assert.isTrue(matcher.find(), "Check SCHEMA definition in " + PY_FACTS_PRECHARGER_NAME);
        String schema = matcher.group(1);

        matcher = Pattern.compile("\"name\": \"(.*?)\"").matcher(schema);

        ListF<String> foundNames = Cf.arrayList();

        while (matcher.find()) {
            foundNames.add(matcher.group(1));
        }

        fields = foundNames.toList();

        String mrScript = getPythonScript(PY_PRECHARG_FACTS_MR_NAME);

        matcher = Pattern.compile("VALID_TYPES = \\[(.*?)\\]").matcher(mrScript);
        Assert.isTrue(matcher.find(), "Check VALID_TYPES definition in " + PY_PRECHARG_FACTS_MR_NAME);
        valid_types = Cf.x(matcher.group(1).split(", ?")).map(type -> type.substring(1, type.length() - 1)).toList();
    }

    private String getPythonScript(String fileName) {
        try {
            return IOUtils.toString(new ClassPathResourceInputStreamSource(this.getClass(), fileName).getInput());
        } catch (IOException e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            return "";
        }
    }


    private String toSqlInArrayCondition(ListF<?> items) {
        return "IN (" + items.mkString("'", "', '", "', ") + ")";
    }

    private String getLatestNotEmptyTableByNow(YPath path) {
        return tableNameResolver.getLatestNotEmptyTable(path, LocalDate.now())
                .getOrThrow("No not empty table found at " + path).toString();
    }

    private String getLatestPrechargedName() {
        return getLatestNotEmptyTableByNow(ytPrechargedPath);
    }

    private String getLatestLogName() {
        return getLatestNotEmptyTableByNow(ytLogsPath);
    }

    public YqlResponse run(ListF<Long> uids, String path) {
        Validate.isTrue(path.matches("[^\\]]*"));

        String uidsCondition = toSqlInArrayCondition(uids);
        String typesCondition = toSqlInArrayCondition(valid_types);
        String fieldsString = String.join(", ", fields);

        String query =
                "PRAGMA inferscheme;\n" +
                        "\n" +
                        "USE " + proxy + ";\n" +
                        "\n" +
                        "INSERT INTO [" + path + "]" +
                        "\n" +
                        "SELECT * FROM (\n" +
                        "    (SELECT\n" +
                        "         " + fieldsString + "\n" +
                        "    FROM [" + getLatestPrechargedName() + "]\n" +
                        "    WHERE uid " + uidsCondition + ")\n" +
                        "\n" +
                        "    UNION ALL \n" +
                        "\n" +
                        "    (SELECT \n" +
                        "        " + fieldsString + "\n" +
                        "    FROM [" + getLatestLogName() + "]\n" +
                        "    WHERE uid " + uidsCondition + "\n" +
                        "    AND type " + typesCondition + ")\n" +
                        ") ORDER BY uid;";

        return yql.run(query);
    }

    public YqlResponse getStatus(String id) {
        return yql.poll(id);
    }
}
