package ru.yandex.chemodan.app.persapi.log;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.persapi.fact.Fact;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.chemodan.util.tskv.TskvUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.parse.JacksonJsonNodeWrapper;
import ru.yandex.misc.digest.Sha256;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author tolmalev
 */
public class FactLogUtils {
    public static final String ALL_FACTS_LOG = "axis-all-persfacts-log";

    private static final String UID_KEY = "uid";
    private static final String SOURCE_KEY = "source";
    private static final String TYPE_KEY = "type";
    private static final String UNIXTIME_KEY = "unixtime";
    private static final String EXTRACT_TIME_KEY = "extract_time";
    private static final String DATA_KEY = "data";
    private static final String TSKV_FORMAT_KEY = "tskv_format";

    private static final String UNKNOWN = "unknown";

    public static String getFactLogString(Fact fact) {

        MapF<String, String> data = Cf.hashMap();

        data.put(TSKV_FORMAT_KEY, ALL_FACTS_LOG);
        data.put(UID_KEY, fact.uid.toString());
        data.put(SOURCE_KEY, fact.source);
        data.put(TYPE_KEY, fact.type);
        data.put(UNIXTIME_KEY, (Instant.now().getMillis() / 1000) + "");
        data.put(EXTRACT_TIME_KEY, TimeUtils.ISO_DATE_TIME_FORMATTER
                .withZone(TimeUtils.EUROPE_MOSCOW_TIME_ZONE)
                .print(fact.extractTime));
        data.put(DATA_KEY, fact.data.toString());

        return TskvUtils.formatTskvLine(data);
    }

    public static Option<String> extractUid(MapF<String, String> tskv) {
        return tskv.getO(UID_KEY);
    }

    public static String extractSource(MapF<String, String> tskv) {
        return tskv.getOrElse(SOURCE_KEY, UNKNOWN);
    }

    public static String extractFactType(MapF<String, String> tskv) {
        return tskv.getOrElse(TYPE_KEY, UNKNOWN);
    }

    public static String extractFactType(String line) {
        return extractFactType(TskvUtils.extractTskv(line));
    }

    public static String replaceFormat(String line, String newFormat) {
        return replaceField(line, TSKV_FORMAT_KEY, newFormat);
    }

    public static String replaceField(String line, String field, String newValue) {
        return TskvUtils.formatTskvLine(TskvUtils
                .extractTskv(line)
                .plus1(field, newValue));
    }

    public static String replaceField(String line, String field, Function<String, String> replacer) {
        return TskvUtils.formatTskvLine(replaceField(TskvUtils.extractTskv(line), field, replacer));
    }

    public static MapF<String, String> replaceField(MapF<String, String> values, String field, Function<String, String> replacer) {
        if (values.containsKeyTs(field)) {
            values = values.plus1(field, replacer.apply(values.getTs(field)));
        }
        return values;
    }

    public static Fact parseTskv(String line) {
        MapF<String, String> tskv = TskvUtils.extractTskv(line);

        return new Fact(
                PassportUid.cons(Long.parseLong(tskv.getTs(UID_KEY))),
                tskv.getTs(SOURCE_KEY),
                new DateTime(Long.parseLong(tskv.getTs(EXTRACT_TIME_KEY)), DateTimeZone.UTC),
                tskv.getTs(TYPE_KEY),
                new JacksonJsonNodeWrapper(JsonNodeUtils.getNode(tskv.getTs(DATA_KEY)))
        );
    }

    public static String hashField(String value) {
        return Sha256.A.digest(value).hex();
    }

    public static String hashFields(String line, SetF<String> fieldsToHash) {
        MapF<String, String> values = TskvUtils.extractTskv(line);

        for (String field : fieldsToHash) {
            try {
                if (!field.contains(".")) {
                    values = replaceField(values, field, FactLogUtils::hashField);
                } else {
                    String[] parts = field.split("\\.");
                    field = parts[0];
                    String subfield = parts[1];

                    Option<String> jsonField = values.getO(field);
                    if (jsonField.isPresent()) {
                        try {
                            ObjectNode node = (ObjectNode) JsonNodeUtils.getNode(jsonField.get());

                            if (node.has(subfield) && node.get(subfield).isValueNode()) {
                                node.set(subfield, new TextNode(hashField(node.get(subfield).asText())));

                                values = values.plus1(field, node.toString());
                            }
                        } catch (RuntimeException ignored) {
                            // don't fail on unparsable json
                        }
                    }
                }
            } catch (Throwable e) {
                ExceptionUtils.throwIfUnrecoverable(e);
            }
        }

        return TskvUtils.formatTskvLine(values);
    }
}
