package ru.yandex.chemodan.app.persapi.schema;

import java.util.Iterator;

import com.fasterxml.jackson.databind.JsonNode;
import com.github.fge.jsonschema.core.exceptions.ProcessingException;
import com.github.fge.jsonschema.core.report.LogLevel;
import com.github.fge.jsonschema.core.report.ProcessingMessage;
import com.github.fge.jsonschema.core.report.ProcessingReport;
import com.github.fge.jsonschema.main.JsonSchema;
import com.github.fge.jsonschema.main.JsonSchemaFactory;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.persapi.acl.BunkerDynamicRegistryBase;
import ru.yandex.chemodan.util.exception.SchemaValidationException;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.inside.bunker.BunkerClient;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author tolmalev
 */
public class FactSchemaManager extends BunkerDynamicRegistryBase<JsonSchema> implements MonicaContainer {
    private static final Logger logger = LoggerFactory.getLogger(FactSchemaManager.class);

    @MonicaMetric
    private final InstrumentMap checks = new InstrumentMap();

    public FactSchemaManager(BunkerClient bunkerClient, Duration updatePeriod) {
        super(bunkerClient, updatePeriod, "types");
    }

    public ListF<String> listKnownTypes() {
        return configByNode.keys();
    }

    @Override
    protected JsonSchema parseNode(byte[] bytes) {
        JsonNode jsonNode = JsonNodeUtils.getNode(new String(bytes));
        try {
            return JsonSchemaFactory.byDefault().getJsonSchema(jsonNode);
        } catch (ProcessingException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public void checkSchema(String type, BenderJsonNode data) {
        checks.measure(() -> {
            logger.trace("Check schema type={}, data={}", type, data);
            Option<JsonSchema> schemaO = configByNode.getO(type);
            if (!schemaO.isPresent()) {
                throw new SchemaValidationException("Unknown type '" + type + "'");
            }

            ProcessingReport report = schemaO.get().validateUnchecked(JsonNodeUtils.getNode(data.toString()));

            if (!report.isSuccess()) {
                throw new SchemaValidationException(makeExceptionString(report, type));
            }
        }, new MetricName(type), UpdateMode.RECURSIVE);
    }

    private String makeExceptionString(ProcessingReport report, String type) {
        Iterator<ProcessingMessage> iterator = report.iterator();

        StringBuilder sb = new StringBuilder();
        sb.append(StringUtils.format("[type={}]", type));
        while (iterator.hasNext()) {
            ProcessingMessage processingMessage = iterator.next();
            if (processingMessage.getLogLevel() == LogLevel.ERROR
                    || processingMessage.getLogLevel() == LogLevel.FATAL)
            {
                sb.append("\n");
                sb.append(processingMessage.getMessage());
            }
        }

        return sb.toString();
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName("fact-manager", new MetricName("facts", "schema-manager"), "FactSchemaManager");
    }
}
