package ru.yandex.chemodan.app.factprocessor;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;

import com.google.common.util.concurrent.ListenableFuture;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.persapi.log.FactLogUtils;
import ru.yandex.chemodan.logbroker.PushClientFactory;
import ru.yandex.chemodan.util.ping.PingerChecker;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.inside.bunker.BunkerClient;
import ru.yandex.inside.logbroker.push.PushClient;
import ru.yandex.inside.logbroker.push.results.DataSendResult;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.annotation.MonicaStaticRegistry;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.spring.Service;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author tolmalev
 */
public class FactSplitter extends DelayingWorkerServiceBeanSupport
        implements LogListener, Service, MonicaContainer, PingerChecker
{
    private static final Logger logger = LoggerFactory.getLogger(FactSplitter.class);

    private final BunkerClient bunkerClient;
    private final PushClientFactory pushClientFactory;
    private final String baseNode;

    private MapF<String, ListF<FactPusher>> pushers = Cf.hashMap();

    @MonicaMetric
    private final InstrumentMap stores = new InstrumentMap();
    @MonicaMetric
    private final MeterMap storedBytes = new MeterMap();

    private final AtomicLong storedFactsCount = new AtomicLong(0);
    private final AtomicLong failedFactsCount = new AtomicLong(0);

    private boolean initialized;

    public FactSplitter(BunkerClient bunkerClient, PushClientFactory pushClientFactory) {
        this.bunkerClient = bunkerClient;
        this.pushClientFactory = pushClientFactory;

        setSleepBeforeFirstRun(false);
        setSleepMode("period");
        setDelay(Duration.standardMinutes(1));
        start();

        baseNode = EnvironmentType.getActive() == EnvironmentType.PRODUCTION
            ? "splitter"
            : EnvironmentType.getActive().name().toLowerCase() + "_splitter";
    }

    @Override
    public void processLogLine(String line) {
        try {
            String type = FactLogUtils.extractFactType(line);

            if (!initialized) {
                throw new RuntimeException("Not inited");
            }

            pushers
                    .getOrElse(type, Cf.list())
                    .forEach(pusher -> pusher.sendFact(line));
        } catch (RuntimeException e) {
            logger.error("Failed to process fact: {}", e);
        }
    }

    private void incAndLog(AtomicLong atomicLong, String s) {
        long currentValue = atomicLong.incrementAndGet();
        if (currentValue % 100 == 0) {
            logger.info(s, currentValue);
        }
    }

    @Override
    public void stop() {
        super.stop();
        pushers.values().<FactPusher>flatten().forEach(pusher -> {
            try {
                pusher.stop();
            } catch (Exception e) {
                logger.error("Failed to stop push client: {}", e);
            }
        });
    }

    @Override
    protected void execute() throws Exception {
        try {
            ListF<FactPusherConfig> configs = new ByteArrayInputStreamSource(bunkerClient.get(baseNode))
                    .readLines()
                    .map(StringUtils::trim)
                    .filter(StringUtils::isNotBlank)
                    .map(FactPusherConfig::parse);

            ListF<FactPusher> pushClientsTmp = Cf.x(pushers.values().flatten());

            pushClientsTmp = pushClientsTmp.filter(pusher -> configs.containsTs(pusher.getConfig()));

            ListF<FactPusher> newPushers = configs
                    .unique()
                    .minus(pushClientsTmp.map(FactPusher::getConfig))
                    .map(config -> new FactPusher(pushClientFactory.build(config.logtype), config));

            newPushers.forEach(FactPusher::start);

            //register new push clients in monica
            newPushers
                    .forEach(pusher -> MonicaStaticRegistry.register(pusher.pushClient, pusher.config.logtype));

            pushClientsTmp = pushClientsTmp.plus(newPushers);

            pushers = pushClientsTmp.groupBy(pusher -> pusher.config.factType);

            logger.info("Started splitting for types {}", pushers.keys());
        } catch (RuntimeException e) {
            logger.error("Failed getting nodes from bunker: {}", e);
            throw e;
        }
        logger.info("Loaded config");
        initialized = true;
    }


    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName("fact-splitter", new MetricName("facts", "splitter"), "FactSplitter");
    }

    @Override
    public boolean isActive() {
        return initialized;
    }

    private static final class FactPusherConfig extends DefaultObject {
        private final String factType;
        private final String logtype;
        private final SetF<String> fieldsToHash;

        private FactPusherConfig(String factType, String logtype, SetF<String> fieldsToHash) {
            this.factType = factType;
            this.logtype = logtype;
            this.fieldsToHash = fieldsToHash;
        }

        private static String logTypeForType(String type) {
            return "axis-" + type + "-fact-log";
        }

        public static FactPusherConfig parse(String configLine) {
            if (!configLine.contains(";")) {
                return new FactPusherConfig(configLine, logTypeForType(configLine), Cf.set());
            } else {
                ListF<String> parts = parseStringList(configLine, ";");

                SetF<String> fieldsToHash = parts
                        .getO(2)
                        .map(fieldsStr -> parseStringList(fieldsStr, ",").unique())
                        .getOrElse(Cf.set());

                return new FactPusherConfig(parts.get(0), parts.get(1), fieldsToHash);
            }
        }
    }

    private static ListF<String> parseStringList(String s, String splitter) {
        return Cf.x(s.split(splitter))
                .map(StringUtils::trim)
                .filter(StringUtils::isNotBlank);
    }

    private final class FactPusher implements Service {
        private final PushClient pushClient;
        private final FactPusherConfig config;

        private FactPusher(PushClient pushClient, FactPusherConfig config) {
            this.config = config;
            this.pushClient = pushClient;
        }

        public FactPusherConfig getConfig() {
            return config;
        }

        @Override
        public void start() {
            pushClient.start();
        }

        @Override
        public void stop() {
            pushClient.stop();
        }

        public void sendFact(String line) {
            String type = config.factType;
            line = FactLogUtils.replaceFormat(line, config.logtype);
            line = FactLogUtils.hashFields(line, config.fieldsToHash);

            ListenableFuture<DataSendResult> future = pushClient.sendData(line.getBytes());

            final String finalLine = line;
            stores.measure(() -> {
                try {
                    DataSendResult result = future.get(5000, TimeUnit.MILLISECONDS);
                    incAndLog(storedFactsCount, "Stored total facts: {}");
                    storedBytes.inc(result.bytes, new MetricName(type), UpdateMode.RECURSIVE);

                    logger.trace("Stored splitted fact: {}", finalLine);

                } catch (TimeoutException | InterruptedException | ExecutionException e) {
                    incAndLog(failedFactsCount, "Failed total facts: {}");
                    logger.error("Failed to send splitted fact of type={}: {}", type, e);
                    throw new RuntimeException("Failed push to LogBroker", e);
                }
            }, new MetricName(type), UpdateMode.RECURSIVE);
        }
    }
}
