package ru.yandex.chemodan.app.factprocessor;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;

import com.google.common.util.concurrent.ListenableFuture;

import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.persapi.log.FactLogUtils;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.inside.logbroker.push.PushClient;
import ru.yandex.inside.logbroker.push.results.DataSendResult;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class SimpleFactSplitter implements LogListener {
    private static final Logger logger = LoggerFactory.getLogger(SimpleFactSplitter.class);

    private final AtomicLong storedFactsCount = new AtomicLong(0);
    private final AtomicLong failedFactsCount = new AtomicLong(0);

    private final PushClient pushClient;
    private final Function1B<String> needSplitF;
    private final long pushWaitMillis;

    private void incAndLog(AtomicLong atomicLong, String s) {
        //TODO: move to some common class
        long currentValue = atomicLong.incrementAndGet();
        if (currentValue % 100 == 0) {
            logger.info(s, currentValue);
        }
    }

    public SimpleFactSplitter(PushClient pushClient, Function1B<String> needSplitF, long pushWaitMillis) {
        this.pushClient = pushClient;
        this.needSplitF = needSplitF;
        this.pushWaitMillis = pushWaitMillis;
    }

    @Override
    public void processLogLine(String line) {
        if (needSplitF.apply(line)) {
            line = FactLogUtils.replaceFormat(line, pushClient.getLogType());

            ListenableFuture<DataSendResult> future = pushClient.sendData(line.getBytes());
            try {
                DataSendResult result = future.get(pushWaitMillis, TimeUnit.MILLISECONDS);
                incAndLog(storedFactsCount, "Stored total facts: {}");

                logger.trace("Stored splitted fact: {}", line);

            } catch (TimeoutException | InterruptedException | ExecutionException e) {
                incAndLog(failedFactsCount, "Failed total facts: {}");
                logger.error("Failed to send splitted fact: {}", e);
                throw new RuntimeException("Failed push to LogBroker", e);
            }
        }
    }
}
