package ru.yandex.chemodan.app.factprocessor.bass;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.factprocessor.SimpleFactSplitter;
import ru.yandex.chemodan.app.persapi.acl.AccessType;
import ru.yandex.chemodan.app.persapi.acl.FactAclManager;
import ru.yandex.chemodan.app.persapi.log.FactLogUtils;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.logbroker.LbConsumerSettings2PrefixImpl;
import ru.yandex.chemodan.logbroker.PushClientFactory;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.util.tskv.TskvUtils;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumerAllDcSettingsProvider;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumerSettings2;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumerWorkerService;
import ru.yandex.inside.logbroker.pull.consumer.LbConsumers;
import ru.yandex.inside.logbroker.pull.consumer.LbLineListener;
import ru.yandex.inside.logbroker.pull.consumer.LbSimpleAsyncLineListener;
import ru.yandex.inside.logbroker.pull.consumer.SynchronousThreadPoolUtil;
import ru.yandex.inside.logbroker.pull.model.LbTopicFilter;
import ru.yandex.inside.logbroker.push.PushClient;
import ru.yandex.misc.parse.CommaSeparated;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
public class FactProcessorBassConsumerContextConfiguration {
    @Value("${logbroker.listener.executor.queue-timeout}")
    private Duration listenerQueueTimeout;

    @Value("${logbroker.listener.executor.total-thread-count}")
    private int listenerThreadsTotalCount;

    @Value("${logbroker.idents-to-exclude}")
    private CommaSeparated excludeIdents;

    @Value("${logbroker.pull.logtype}")
    private String logtype;

    @Bean
    public BassClient bassClient(@Value("${bass.host}") String bassHost) {
        return new BassClient(bassHost, bassHttpClientConfigurator().configure());
    }

    @Bean
    @OverridableValuePrefix("bass")
    public HttpClientConfigurator bassHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    protected LbLineListener bassLbConsumerListener(
            @Qualifier("bassFactSplitter")
            SimpleFactSplitter bassFactSplitter)
    {
        return new LbSimpleAsyncLineListener(bassFactSplitter,
                SynchronousThreadPoolUtil.listenerExecutor(listenerQueueTimeout, listenerThreadsTotalCount)
        );
    }

    @Bean
    public PushClient bassPushClient(
            PushClientFactory pushClientFactory,
            @Value("${bass.export.logtype}")
            String logType)
    {
        return pushClientFactory.build(logType);
    }

    @Bean
    protected SimpleFactSplitter bassFactSplitter(
            FactAclManager factAclManager,
            BassClient bassClient,
            @Value("${bass.push.wait}")
            Duration pushWait,
            @Qualifier("bassPushClient")
            PushClient pushClient)
    {
        return new SimpleFactSplitter(pushClient, (line) -> {
            MapF<String, String> tskv = TskvUtils.extractTskv(line);
            String source = FactLogUtils.extractSource(tskv);
            String type = FactLogUtils.extractFactType(tskv);

            if (!factAclManager.isAccessAllowed("bass", source, type, AccessType.READ)) {
                // bass doesn't have access to this fact
                return false;
            }

            // check if bass knows this uid
            Option<String> uidO = FactLogUtils.extractUid(tskv);
            ListF<String> goodUids = bassClient.checkUids(uidO);

            if (goodUids.containsTs(uidO.get())) {
                return true;
            }

            return false;
        }, pushWait.getMillis());
    }

    @Bean
    @OverridableValuePrefix("facts-bass")
    protected LbConsumerSettings2 bassLbConsumerSettings() {
        return new LbConsumerSettings2PrefixImpl("bass",
                new LbTopicFilter()
                        .withIdentsToExclude(excludeIdents.list)
                        .withLogType(logtype)
        );
    }

    @Bean
    public LbConsumerWorkerService bassLbConsumerWorkerService(
            @Value("${logbroker.pull.consumer-count}")
            int consumerCount,
            @Qualifier("bassLbConsumerListener")
            LbLineListener bassLbConsumerListener,
            @Qualifier("bassLbConsumerSettings")
            LbConsumerSettings2 bassLbConsumerSettings)
    {
        return new LbConsumerWorkerService(
                new LbConsumers(new LbConsumerAllDcSettingsProvider(bassLbConsumerSettings), bassLbConsumerListener)
                        .fixedCount(consumerCount)
        );
    }
}
