package ru.yandex.chemodan.app.balancer;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.balancer.selection.UrlSelectionStrategy;
import ru.yandex.chemodan.ping.HostsManager;
import ru.yandex.chemodan.uploader.status.strategy.LoadingStatusStrategyType;
import ru.yandex.misc.test.Assert;

import static org.mockito.Mockito.when;

/**
 * @author nshmakov
 */
public class BalancerImplTest {

    private BalancerImpl sut = new BalancerImpl(3);

    private ListF<String> threeHosts = Cf.list("host1", "host2", "host3");
    private MapF<String, Long> hostWithSummarizedStatus = Cf.map("host1", 2L).plus1("host2", 1L).plus1("host3", 3L);
    private MapF<String, MapF<LoadingStatusStrategyType, Long>> hostsStatus =
            Cf.<String, MapF<LoadingStatusStrategyType, Long>>map()
            .plus1("host1", Cf.map(LoadingStatusStrategyType.LA, 2L))
            .plus1("host2", Cf.map(LoadingStatusStrategyType.LA, 1L))
            .plus1("host3", Cf.map(LoadingStatusStrategyType.LA, 3L));
    private String slbUrl = "slb";

    @Mock
    private UrlSelectionStrategy selectionStrategyMock;
    @Mock
    private HostsManager<MapF<LoadingStatusStrategyType, Long>> hostsManagerMock;
    @Mock
    private LoadingStatusSummarizingStrategy loadingStatusMergingStrategyMock;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
        sut.setSelectionStrategy(selectionStrategyMock);
        sut.setHostsManager(hostsManagerMock);
        sut.setSlbUrl(slbUrl);
        sut.setLoadingStatusSummarizingStrategy(loadingStatusMergingStrategyMock);
    }

    @Test
    public void shouldSelectUploaderHost() {
        when(hostsManagerMock.getAvailableHostsResults()).thenReturn(hostsStatus);
        when(loadingStatusMergingStrategyMock.summarizeLoadingStatus(hostsStatus)).thenReturn(hostWithSummarizedStatus);
        when(selectionStrategyMock.select(Cf.list("host2", "host1", "host3"))).thenReturn("host1");

        String actual = sut.selectUploader();

        Assert.equals("host1", actual);
    }

    @Test
    public void shouldCutOffHostsIfSelectionLimitIsLesserThanHostsCount() {
        when(hostsManagerMock.getAvailableHostsResults()).thenReturn(hostsStatus);
        when(loadingStatusMergingStrategyMock.summarizeLoadingStatus(hostsStatus)).thenReturn(
                Cf.map("host1", 1L)
                        .plus1("host2", 2L)
                        .plus1("host3", 3L)
                        .plus1("host4", 4L)
                        .plus1("host5", 5L)
        );
        when(selectionStrategyMock.select(threeHosts)).thenReturn("host2");

        String actual = sut.selectUploader();

        Assert.equals("host2", actual);
    }

    @Test
    public void shouldSelectSlbHostWhenUploadersIsUnavailable() {
        when(hostsManagerMock.getAvailableHostsResults()).thenReturn(Cf.<String, MapF<LoadingStatusStrategyType, Long>>map());

        String actual = sut.selectUploader();

        Assert.equals(slbUrl, actual);
    }

    @Test
    public void shouldReturnUploadersLoadingStatus() {
        when(hostsManagerMock.getAvailableHostsResults()).thenReturn(hostsStatus);

        MapF<String, MapF<LoadingStatusStrategyType, Long>> actual = sut.getUploadersLoadingStatus();

        Assert.equals(hostsStatus, actual);
    }
}
