package ru.yandex.chemodan.app.balancer;

import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.uploader.status.strategy.LoadingStatusStrategyType;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author nshmakov
 */
public class LoadingStatusSummarizingStrategy {

    private static final Logger logger = LoggerFactory.getLogger(LoadingStatusSummarizingStrategy.class);

    @Autowired
    private LoadingStatusCoefficientsHolder loadingStatusCoefficientHolder;
    @Autowired
    private LoadingStatusTypesHolder loadingStatusTypesHolder;

    public MapF<String, Long> summarizeLoadingStatus(MapF<String, MapF<LoadingStatusStrategyType, Long>> hostsStatus) {
        MapF<String, Long> result = Cf.hashMap();
        for (String host : hostsStatus.keySet()) {
            long sum = loadingStatusTypesHolder.getTypes().stream()
                    .map(type -> (long) (getCoefficientForType(type) * hostsStatus.getTs(host).getTs(type)))
                    .collect(Collectors.summingLong(Long::valueOf));
            result.put(host, sum);
        }
        return result;
    }

    private double getCoefficientForType(LoadingStatusStrategyType type) {
        switch (type) {
            case LA:
                return loadingStatusCoefficientHolder.getLaCoefficient();
            case CONTENT:
                return loadingStatusCoefficientHolder.getContentCoefficient();
            case DISK_IO:
                return loadingStatusCoefficientHolder.getDiskIoCoefficient();
            case DISK_IOPS:
                return loadingStatusCoefficientHolder.getDiskIopsCoefficient();
            case NETWORK_IO:
                return loadingStatusCoefficientHolder.getNetworkIoCoefficient();
            case QUEUE_SIZE_USER:
                return loadingStatusCoefficientHolder.getQueueSizeUserCoefficient();
            case EXP_USER_QUEUE_INITIAL:
                return loadingStatusCoefficientHolder.getExpUserQueueInitialCoefficient();
            case EXP_USER_QUEUE_IN_PROGRESS:
                return loadingStatusCoefficientHolder.getExpUserQueueInProgressCoefficient();
            case QUEUE_SIZE_MULCA:
                return loadingStatusCoefficientHolder.getQueueSizeMulcaCoefficient();
            default:
                logger.warn("Unknown LoadingStatusStrategyType: {}. Set default coefficient for this type (=1)", type);
                return 1f;
        }
    }

    public void setLoadingStatusCoefficientHolder(LoadingStatusCoefficientsHolder loadingStatusCoefficientHolder) {
        this.loadingStatusCoefficientHolder = loadingStatusCoefficientHolder;
    }

    public void setLoadingStatusTypesHolder(LoadingStatusTypesHolder loadingStatusTypesHolder) {
        this.loadingStatusTypesHolder = loadingStatusTypesHolder;
    }
}
