package ru.yandex.chemodan.app.balancer.client;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.MapLikeType;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.balancer.exception.UploaderNotAvailableException;
import ru.yandex.chemodan.ping.PingClient;
import ru.yandex.chemodan.uploader.status.strategy.LoadingStatusStrategyType;
import ru.yandex.chemodan.uploader.web.ApiUrls;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadStringResponseHandler;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author nshmakov
 */
public class UploaderClient implements PingClient<MapF<LoadingStatusStrategyType, Long>> {

    private static final Logger logger = LoggerFactory.getLogger(UploaderClient.class);

    @Value("${control.http.port}")
    private int port;

    private final HttpClient httpClient;

    private final ObjectMapper objectMapper = new ObjectMapper();
    private final MapLikeType loadingStatusResponseType = objectMapper.getTypeFactory()
            .constructMapLikeType(HashMap.class, LoadingStatusStrategyType.class, Long.class);

    public UploaderClient(HttpClient httpClient) {
        this.httpClient = httpClient;
    }

    /**
     * @throws ru.yandex.chemodan.app.balancer.exception.UploaderNotAvailableException when Uploader is not available
     */
    @Override
    public MapF<LoadingStatusStrategyType, Long> ping(String host) {
        try {
            String url = "http://" + host + ":" + port + ApiUrls.LOADING_STATUS;
            final String response = ApacheHttpClientUtils.execute(new HttpGet(url), httpClient, new ReadStringResponseHandler());
            logger.debug("Uploader {} result {}", host, response);
            return statusFromJson(response);
        } catch (Exception ex) {
            throw new UploaderNotAvailableException(ex);
        }
    }

    private MapF<LoadingStatusStrategyType, Long> statusFromJson(String json) {
        try {
            Map<LoadingStatusStrategyType, Long> result = objectMapper.readValue(json, loadingStatusResponseType);
            return Cf.wrap(result);
        } catch (IOException ex) {
            throw ExceptionUtils.translate(ex);
        }
    }

    public void setPort(int port) {
        this.port = port;
    }
}
