package ru.yandex.chemodan.app.balancer.servlet;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicNameValuePair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.chemodan.app.balancer.Balancer;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpRequestUtils;

/**
 * @author nshmakov
 */
public class HttpProxyServlet extends HttpServlet {

    private static final Logger logger = LoggerFactory.getLogger(HttpProxyServlet.class);

    @Autowired
    private Balancer balancer;
    @Value("${control.http.port}")
    private int port;

    @Autowired
    @Qualifier("proxyServletHttpClient")
    private HttpClient httpClient;

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        logger.info(HttpRequestUtils.describeRemoveSensitive(req));

        String targetUrl = getTargetUrl(req);
        HttpPost forwardRequest = createProxyPostRequest(targetUrl, req);
        forwardRequestTo(forwardRequest, resp, targetUrl);
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        logger.info(HttpRequestUtils.describeRemoveSensitive(req));

        String targetUrl = getTargetUrl(req);
        HttpGet forwardRequest = createProxyGetRequest(targetUrl, req);
        forwardRequestTo(forwardRequest, resp, targetUrl);
    }

    private String getTargetUrl(HttpServletRequest req) {
        return "http://" + balancer.selectUploader() + ":" + port + req.getRequestURI();
    }

    private HttpGet createProxyGetRequest(String targetUrl, final HttpServletRequest req) {
        Map<String, Object> params = new HashMap<>(req.getParameterMap().size());
        req.getParameterMap().forEach((key, value) -> params.put(key, value[0]));
        String urlWithParams = UrlUtils.addParameters(targetUrl, params);
        return new HttpGet(urlWithParams);
    }

    private void forwardRequestTo(HttpRequestBase forwardRequest, HttpServletResponse resp, String targetUrl) {
        logger.info("Forward request to {}", targetUrl);
        try {
            HttpResponse targetResponse = httpClient.execute(forwardRequest);
            copyHeaders(targetResponse.getAllHeaders(), resp);
            resp.setStatus(targetResponse.getStatusLine().getStatusCode());

            HttpEntity entity = targetResponse.getEntity();
            if (entity != null) {
                entity.writeTo(resp.getOutputStream());
            }
        } catch (Exception ex) {
            throw ExceptionUtils.translate(ex);
        } finally {
            forwardRequest.releaseConnection();
        }
    }

    private HttpPost createProxyPostRequest(String targetUrl, HttpServletRequest req) throws UnsupportedEncodingException {
        HttpPost post = new HttpPost(targetUrl);
        post.setHeaders(getHeaders(req));
        post.setEntity(new UrlEncodedFormEntity(getParameters(req), "UTF-8"));
        return post;
    }

    private List<NameValuePair> getParameters(final HttpServletRequest req) {
        return Collections.list(req.getParameterNames()).stream()
                .map(name -> new BasicNameValuePair(name, req.getParameter(name)))
                .collect(Collectors.toList());
    }

    private Header[] getHeaders(final HttpServletRequest req) {
        return Collections.list(req.getHeaderNames()).stream()
                .filter(name -> !name.equals(HttpHeaders.CONTENT_LENGTH))
                .map(name -> new BasicHeader(name, req.getHeader(name)))
                .collect(Collectors.toList())
                .toArray(new Header[0]);
    }

    private void copyHeaders(Header[] headers, final HttpServletResponse to) {
        Arrays.asList(headers).forEach(header -> to.setHeader(header.getName(), header.getValue()));
    }
}
