package ru.yandex.chemodan.app.balancer.servlet;

import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.message.BasicNameValuePair;
import org.joda.time.Duration;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.balancer.Balancer;
import ru.yandex.chemodan.app.balancer.config.BalancerDaemonContextConfiguration;
import ru.yandex.chemodan.app.balancer.config.ServletContextConfiguration;
import ru.yandex.chemodan.boot.ChemodanInitContextConfiguration;
import ru.yandex.chemodan.http.CommonHeaders;
import ru.yandex.chemodan.test.TestManager;
import ru.yandex.chemodan.uploader.registry.ApiVersion;
import ru.yandex.chemodan.uploader.web.ApiArgs;
import ru.yandex.chemodan.uploader.web.ApiUrls;
import ru.yandex.chemodan.util.test.AbstractTest;
import ru.yandex.devtools.test.annotations.YaExternal;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;

import static org.mockito.Mockito.reset;
import static org.mockito.Mockito.when;

/**
 * @author nshmakov
 */
@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration(classes = {
        ChemodanInitContextConfiguration.class,
        ServletContextConfiguration.class,
        HttpProxyServletTest.Config.class
})
@YaExternal
public class HttpProxyServletTest extends AbstractTest {

    @Value("${uploader-balancer.http.port}")
    private int balancerPort;
    @Value("${mulca.gate.url}")
    private String mulcagateUrl;
    @Value("${mulca.service}")
    private String mulcaService;
    @Value("${mulca.namespace}")
    private String mulcaNamespace;
    @Value("${uploader-balancer.socket.maxIdleTime}")
    private Duration maxIdleTime;
    private String uploaderHost;
    private String balancerUrl;

    @Autowired
    private Balancer balancerMock;
    @Autowired
    private HttpProxyServlet proxyServlet;

    private MulcaClient mulcaClient;

    private SingleWarJetty balancerJetty;

    @Before
    public void init() {
        reset(balancerMock);

        balancerJetty = new BalancerDaemonContextConfiguration().balancerJetty(
                10, 10, balancerPort, 50, maxIdleTime, proxyServlet);
        balancerJetty.start();

        uploaderHost = "uploader3h.dst.yandex.net";
        balancerUrl = "http://localhost:" + balancerPort;

        HttpClient mulcaHttpClient = ApacheHttpClientUtils.singleConnectionClient(Timeout.seconds(10));
        mulcaClient = MulcaClient.custom(mulcaHttpClient, mulcagateUrl).withService(mulcaService)
                .withNamespace(mulcaNamespace).build();
    }

    @After
    public void destroy() {
        if (balancerJetty != null) {
            balancerJetty.stop();
        }
    }

    @Test
    public void shouldProxyPostRequest() throws Exception {
        when(balancerMock.selectUploader()).thenReturn(uploaderHost);

        String actual = post(balancerUrl + ApiUrls.UPLOAD_URL + "/disk",
                Cf.map(ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString())
                .plus1(ApiArgs.UID, "1")
                .plus1(ApiArgs.FILE_ID, "tmpfile")
                .plus1(ApiArgs.PATH, "/tmp")
        );

        Assert.notEmpty(actual);
    }

    @Test
    public void shouldProxyIso88591Symbols() throws Exception {
        when(balancerMock.selectUploader()).thenReturn(uploaderHost);

        // will be illegal if interpret it as UTF-8 after UTF-8 -> ISO-8859-1 conversion
        String pathWithIllegalUtf8Symbols
                = new String(new byte[]{ (byte) 0x61, (byte) 0xc3, (byte) 0x80 });

        post(balancerUrl + ApiUrls.UPLOAD_URL + "/disk",
                Cf.map(ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString())
                        .plus1(ApiArgs.UID, "1")
                        .plus1(ApiArgs.FILE_ID, "tmpfile.jpg")
                        .plus1(ApiArgs.PATH, pathWithIllegalUtf8Symbols),
                "UTF-8"
        );
    }

    @Test
    public void shouldProxyGetRequest() {
        when(balancerMock.selectUploader()).thenReturn(uploaderHost);

        TestManager.withEmptyTemporaryFile("get-proxy",
            tmpFile -> {
                TestManager.copyISStoFile2(new ClassPathResourceInputStreamSource(HttpProxyServletTest.class, "test.pdf"), tmpFile);
                MulcaId mulcaId = mulcaClient.upload(tmpFile, "tmp");

                String result = get(balancerUrl + ApiUrls.REGENERATE_PREVIEW_URL,
                        Cf.<String, Object>map()
                                .plus1(ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString())
                                .plus1(ApiArgs.MULCA_ID, mulcaId.getStidCheckNoPart())
                                .plus1(ApiArgs.CONTENT_TYPE, "application/pdf")
                                .plus1(ApiArgs.FILE_SIZE, tmpFile.length())
                                .plus1(ApiArgs.FILE_NAME, "test"));

                Assert.notEmpty(result);
            });
    }

    private String get(String url, MapF<String, Object> params) {
        String urlWithParams = UrlUtils.addParameters(url, params);
        HttpGet request = new HttpGet(urlWithParams);
        request.addHeader(CommonHeaders.YANDEX_CLOUD_REQUEST_ID, "tst-" + Random2.R.nextAlnum(6));
        return ApacheHttpClientUtils.executeReadString(request);
    }

    private String post(String url, MapF<String, String> formParams) throws Exception {
        return post(url, formParams, null);
    }

    private String post(String url, MapF<String, String> formParams, String charset) throws Exception {
        HttpPost request = new HttpPost(url);
        request.setEntity(new UrlEncodedFormEntity(
                formParams.entries().map((key, value) -> new BasicNameValuePair(key, value)),
                charset)
        );

        return ApacheHttpClientUtils.executeReadString(request);
    }

    @Configuration
    public static class Config {

        @Bean
        public Balancer balancer() {
            return Mockito.mock(Balancer.class);
        }

        @Bean
        public HttpClient proxyServletHttpClient() {
            return ApacheHttpClientUtils.singleConnectionClient(Timeout.seconds(10));
        }
    }
}
