package ru.yandex.chemodan.app.bazingaload;

import org.joda.time.Duration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.bazingaload.tasks.CpuLoadTask;
import ru.yandex.chemodan.app.bazingaload.tasks.MultiplicationTask;
import ru.yandex.chemodan.app.bazingaload.tasks.QuickMovePostProcessTask;
import ru.yandex.chemodan.app.bazingaload.tasks.SleepTask;
import ru.yandex.chemodan.bazinga.BazingaControllerConfigurationContextConfiguration;
import ru.yandex.chemodan.bazinga.BazingaWorkerTaskQueues;
import ru.yandex.chemodan.bazinga.ChemodanBazingaWorkerContextConfiguration;
import ru.yandex.chemodan.boot.ChemodanCommonContextConfiguration;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.pg.context.PgBazingaContextConfiguration;
import ru.yandex.commune.bazinga.pg.storage.WorkerIdCondition;
import ru.yandex.commune.bazinga.pg.storage.maintenance.CleanJobsConfiguration;
import ru.yandex.commune.bazinga.pg.storage.maintenance.PgBazingaMaintenanceConfiguration;
import ru.yandex.commune.bazinga.pg.storage.maintenance.RestoreJobsConfiguration;
import ru.yandex.commune.bazinga.pg.storage.maintenance.TaskCleanupSettings;
import ru.yandex.commune.bazinga.pg.storage.maintenance.TaskCleanupSettingsProvider;
import ru.yandex.commune.bazinga.pg.storage.maintenance.UpdateJobCountersConfiguration;
import ru.yandex.commune.bazinga.pg.worker.PgBazingaWorkerConfiguration;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.dynproperties.DynamicProperty;

@Configuration
@Import({
        ChemodanBazingaWorkerContextConfiguration.class,
        PgBazingaContextConfiguration.class,
        ChemodanCommonContextConfiguration.class,
        BazingaControllerConfigurationContextConfiguration.class,
})
public class BazingaLoadContextConfiguration {

    private final DynamicProperty<Integer> cleanJobsBatchSize = new DynamicProperty<>(
            "bazinga.jobs-cleaning.chunk-size", 1000);
    private final DynamicProperty<Integer> storeCompletedJobsHours = new DynamicProperty<>(
            "bazinga.store-in-hours.completed-jobs", 1);
    private final DynamicProperty<Integer> storeFailedJobsHours = new DynamicProperty<>(
            "bazinga.store-in-hours.failed-jobs", 48);
    private final DynamicProperty<Integer> storeCronJobsHours = new DynamicProperty<>(
            "bazinga.store-in-hours.cron-jobs", 96);


    private final DynamicProperty<Integer> restoreAttemptsToConnectWorker = new DynamicProperty<>(
            "bazinga.restore-started-jobs.attempts-to-connect-worker", 3);

    private final DynamicProperty<Integer> restoreAttemptIntervalSeconds = new DynamicProperty<>(
            "bazinga.restore-started-jobs.attempt-interval-seconds", 60);

    private final DynamicProperty<Integer> restoreStartedBeforeSeconds = new DynamicProperty<>(
            "bazinga.restore-started-jobs.started-before-seconds", 2 * 60);


    @Bean
    public PgBazingaWorkerConfiguration pgBazingaWorkerConfiguration() {
        return new PgBazingaWorkerConfiguration(
                Cf.list(TaskQueueName.CPU_INTENSIVE));
    }

    @Bean
    public BazingaWorkerTaskQueues taskQueues() {
        return new BazingaWorkerTaskQueues(
                TaskQueueName.CRON, TaskQueueName.REGULAR, TaskQueueName.CPU_INTENSIVE,
                Cf.list());
    }

    @Bean
    public TasksManager tasksManager(BazingaTaskManager bazingaTaskManager) {
        return new TasksManager(bazingaTaskManager);
    }


    @Bean
    public SleepTask sleepTask(TasksManager tasksManager) {
        return new SleepTask(tasksManager);
    }

    @Bean
    public CpuLoadTask cpuLoadTask(TasksManager tasksManager) {
        return new CpuLoadTask(tasksManager);
    }

    @Bean
    public MultiplicationTask multiplicationTask(TasksManager tasksManager) {
        return new MultiplicationTask(tasksManager);
    }

    @Bean
    public QuickMovePostProcessTask quickMovePostProcessTask() {
        return new QuickMovePostProcessTask();
    }


    @Bean
    public PgBazingaMaintenanceConfiguration bazingaMaintenanceConfiguration() {
        return new PgBazingaMaintenanceConfiguration(
                new CleanJobsConfiguration(cleanJobsBatchSize::get,
                        () -> Duration.standardHours(storeCronJobsHours.get())),
                new RestoreJobsConfiguration(restoreAttemptsToConnectWorker::get,
                        () -> Duration.standardSeconds(restoreAttemptIntervalSeconds.get()),
                        () -> Duration.standardSeconds(restoreStartedBeforeSeconds.get()),
                        WorkerIdCondition.portEq(28702)),
                UpdateJobCountersConfiguration.DEFAULT,
                Option.empty());
    }

    @Bean
    public TaskCleanupSettingsProvider cleanupSettingsProvider() {
        return (taskId -> new TaskCleanupSettings(taskId,
                Duration.standardHours(storeCompletedJobsHours.get()),
                Duration.standardHours(storeFailedJobsHours.get())));
    }


}
