#!/usr/bin/env python

import hashlib
import json
import logging
import uuid

import msgpack
from cocaine.decorators.http_dec import tornado_http
from cocaine.detail.trace import Trace
from cocaine.logger import CocaineHandler, LoggerWithExtraInRecord
from cocaine.services import Service
from cocaine.worker import Worker
from tornado import gen

unistorage = Service("unistorage")
DEFAULT_HEADERS = [('Content-type', 'application/json'), ]
DEFAULT_TIMEOUT = 60
logger = LoggerWithExtraInRecord('')
cocaine_handler = CocaineHandler()
cocaine_handler.setLevel(logging.INFO)
logger.addHandler(cocaine_handler)
logger.setLevel(logging.DEBUG)


class NotFound(Exception):
    pass


def check_error(data):
    if data[0] != 'error':
        return
    body = data[1]
    code = body[0][1]
    message = body[1]
    if code == 4 or "invalid key" in message:
        raise KeyError(message)
    if code == 6 or message == "no such file":
        raise NotFound(message)


def get_logger(trace):
    return logging.LoggerAdapter(logger, extra={'traceid': trace})


@gen.coroutine
def calc_hashes(stid, chunk_timeout, trace):
    offset = 0
    size = 0  # Hint: size=0 - whole file
    chunk_size = 25 * 1024 * 1024  # 1MB
    hash_md5 = hashlib.md5()
    hash_sha256 = hashlib.sha256()
    size_sum = 0

    try:
        channel = yield unistorage.read2(
            stid,
            offset,
            size,
            chunk_size,
            trace=Trace(traceid=trace, parentid=trace, spanid=trace),
        )
        yield channel.tx.meta()
        data = yield channel.rx.get(timeout=chunk_timeout)
        check_error(data)

        if data[0] != 'meta':
            raise Exception('Bad response from unistorage: %s' % json.dumps(data))

        while True:
            yield channel.tx.next()
            response = yield channel.rx.get(timeout=chunk_timeout)
            response_type, response_body = response

            if response_type == 'error':
                error_category, error_number_in_category = response_body[0]
                message = response_body[1] if len(response_body) == 2 else 'unknown error'
                raise Exception('Unistorage error {}: {}'.format(error_number_in_category, message))
            elif response_type == 'chunk':
                # Chunk body format: [data]
                data = response_body[0]
                hash_md5.update(data)
                hash_sha256.update(data)
                size_sum += len(data)
            elif response_type == 'end':
                break
            else:
                raise Exception('Failed to parse unistorage response: "{}"'.format(response))
    finally:
        yield channel.tx.close()

    raise gen.Return({
        'md5': hash_md5.hexdigest(),
        'sha256': hash_sha256.hexdigest(),
        'size': size_sum,
    })


@tornado_http
def calc_hashes_handler(request, response):
    try:
        req = yield request.read()
        headers = req.headers
        args = req.arguments
        trace_id = int(headers.get('X-Request-Id', uuid.uuid4().hex), 16)
        log = get_logger(trace_id)
        stid = args.get('stid', [None])[0]
        if not stid:
            log.info('Bad arguments: %s', args)
            raise KeyError('no stid arg')

        status_code = 200
        result = yield calc_hashes(stid, DEFAULT_TIMEOUT, trace_id)
        log.debug('%s calculated: %s', stid, result)
    except KeyError as e:
        status_code = 400
        result = {'error': e.message}
    except NotFound as e:
        status_code = 404
        result = {'error': e.message}
    except Exception as e:
        status_code = 500
        result = {'error': e.message}
        log.error(e)

    response.write(msgpack.packb((status_code, DEFAULT_HEADERS)))
    response.write(json.dumps(result))
    response.close()


def main():
    w = Worker()
    w.run({"http": calc_hashes_handler})


if __name__ == "__main__":
    main()
