package ru.yandex.chemodan.app.countersapi;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ConsistencyLevel;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.PoolingOptions;
import com.datastax.driver.core.QueryOptions;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.policies.LoadBalancingPolicy;
import com.datastax.driver.core.policies.TokenAwarePolicy;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.countersapi.dao.PublicPageViewsDao;
import ru.yandex.commune.alive2.location.LocationResolver;
import ru.yandex.commune.cassandra.CassandraUtils;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author nshmakov
 */
@Configuration
public class CassandraContextConfiguration {

    private final static Logger logger = LoggerFactory.getLogger(CassandraContextConfiguration.class);

    @Value("${cassandra.conductorGroupName}")
    private String cassandraConductorGroup;
    @Value("${cassandra.consistencyLevel}")
    private ConsistencyLevel consistencyLevel;
    @Value("${cassandra.maxRequestsPerConnection}")
    private int maxRequestsPerConnection;
    @Value("${cassandra.coreConnections}")
    private int coreConnections;
    @Value("${cassandra.maxConnections}")
    private int maxConnections;

    @Autowired
    private LocationResolver locationResolver;

    @Bean(destroyMethod = "close")
    public Cluster cassandraCluster(EnvironmentType environmentType) {
        Option<String> currentDc = Option.empty();

        if (environmentType != EnvironmentType.DEVELOPMENT && environmentType != EnvironmentType.TESTS) {
            currentDc = locationResolver.resolveLocation().dcName;
        }

        ListF<String> cassandraHosts = locationResolver.getGroupHosts(cassandraConductorGroup);

        MapF<String, Option<String>> hostToDcMap = cassandraHosts
                .toMapMappingToValue(host -> locationResolver.resolveLocationFor(Option.of(host)).dcName);

        logger.info("Current DC: {}, hostToDcMap: {}", currentDc, hostToDcMap);

        LoadBalancingPolicy policy = CassandraUtils.createLoadBalancingPolicy(currentDc, hostToDcMap);

        Cluster cluster = Cluster.builder()
                .addContactPoints(cassandraHosts.toArray(String.class))
                .withQueryOptions(new QueryOptions().setConsistencyLevel(consistencyLevel))
                .withPoolingOptions(new PoolingOptions()
                        .setMaxRequestsPerConnection(HostDistance.REMOTE, maxRequestsPerConnection)
                        .setMaxConnectionsPerHost(HostDistance.REMOTE, maxConnections)
                        .setCoreConnectionsPerHost(HostDistance.REMOTE, coreConnections))
                .withLoadBalancingPolicy(new TokenAwarePolicy(policy))
                .build();
        return cluster;
    }

    @Bean
    public Session cassandraSession(EnvironmentType environmentType) {
        return cassandraCluster(environmentType).newSession().init();
    }

    @Bean
    public PublicPageViewsDao publicPageViewsDao() {
        return new PublicPageViewsDao();
    }
}
