package ru.yandex.chemodan.app.countersapi.dao;

import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Instrument;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.time.Stopwatch;

/**
 * @author nshmakov
 */
public class PublicPageViewsDao implements MonicaContainer {

    private static final Duration LONG_RESPONSE_TIME = Duration.millis(300);

    private static final Logger log = LoggerFactory.getLogger(PublicPageViewsDao.class);

    @MonicaMetric
    @GroupByDefault
    private final Instrument increase = new Instrument();
    @MonicaMetric
    @GroupByDefault
    private final Instrument read = new Instrument();

    @Autowired
    private Session cassandraSession;

    public void increaseCounter(final String hash, final long times) {
        increase.measure(() -> {
            Stopwatch stopwatch = Stopwatch.createAndStart();

            ResultSet result = cassandraSession.execute(
                    "UPDATE disk_ks.page_view_counts" +
                            " SET counter_value = counter_value + ?" +
                            " WHERE private_hash = ?",
                    times, hash);

            String message = "Cassandra write " + result.getExecutionInfo().getTriedHosts();
            stopwatch.stopAndLog(message, LONG_RESPONSE_TIME, PublicPageViewsDao.log);
        });
    }

    public MapF<String, Long> readCounters(final ListF<String> hash) {
        return read.measure(() -> {
            Stopwatch stopwatch = Stopwatch.createAndStart();

            String inClause = CqlUtils.generateInClause(hash.size());
            String query = "SELECT private_hash, counter_value FROM disk_ks.page_view_counts " +
                    "WHERE private_hash IN " + inClause;

            ResultSet queryResult = cassandraSession.execute(query, hash.toArray());
            MapF<String, Long> result = Cf.hashMap();
            for (Row row : queryResult) {
                result.put(row.getString("private_hash"), row.getLong("counter_value"));
            }

            String message = "Cassandra read " + queryResult.getExecutionInfo().getTriedHosts();
            stopwatch.stopAndLog(message, LONG_RESPONSE_TIME, log);

            return result;
        });
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "cassandra",
                new MetricName("cassandra", "public-views"),
                "Public views dao"
        );
    }
}
