package ru.yandex.chemodan.app.countersapi.dao;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.countersapi.CassandraContextConfiguration;
import ru.yandex.chemodan.boot.ChemodanInitContextConfiguration;
import ru.yandex.chemodan.boot.ChemodanPropertiesLoadStrategy;
import ru.yandex.chemodan.util.test.AbstractTest;
import ru.yandex.misc.property.load.PropertiesLoader;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.version.SimpleAppName;

import static ru.yandex.chemodan.app.countersapi.dao.PublicPageViewsDaoTestUtils.resetCounterValue;

/**
 * @author nshmakov
 */
@ContextConfiguration(classes = { CassandraContextConfiguration.class, ChemodanInitContextConfiguration.class })
@RunWith(SpringJUnit4ClassRunner.class)
public class PublicPageViewsDaoTest extends AbstractTest {

    @Autowired
    private PublicPageViewsDao publicPageViewsDao;

    private String hash = hash();

    @BeforeClass
    public static void init() {
        PropertiesLoader.initialize(
                new ChemodanPropertiesLoadStrategy(new SimpleAppName("disk", "log-reader"), true));
    }

    @Before
    public void resetCounterForHash() {
        resetCounterValue(publicPageViewsDao, hash);
    }

    @Test
    public void shouldReturnEmptyMapForNonexistentCounter() {
        Assert.isTrue(publicPageViewsDao.readCounters(Cf.list("000")).isEmpty());
    }

    @Test
    public void shouldIncreaseCounter() {
        long times = 3;

        incrementCounter(hash, times);

        Assert.equals(times, publicPageViewsDao.readCounters(Cf.list(hash)).getTs(hash));
    }

    @Test
    public void shouldDecreaseCounter() {
        publicPageViewsDao.increaseCounter(hash, 5);
        publicPageViewsDao.increaseCounter(hash, -3);

        Assert.equals(2L, publicPageViewsDao.readCounters(Cf.list(hash)).getTs(hash));
    }

    @Test
    public void shouldReturnValuesForMultipleHashes() {
        // given
        String hashOne = hash();
        long hashOneCounter = 5;
        incrementCounter(hashOne, hashOneCounter);

        String hashTwo = hash();
        long hashTwoCounter = 10;
        incrementCounter(hashTwo, hashTwoCounter);

        MapF<String, Long> expected = Cf.map(hashOne, hashOneCounter, hashTwo, hashTwoCounter);

        // when
        MapF<String, Long> actual = publicPageViewsDao.readCounters(Cf.list(hashOne, hashTwo));

        // then
        Assert.equals(expected, actual);
    }

    private void incrementCounter(String hash, long times) {
        for (int i = 0; i < times; i++) {
            publicPageViewsDao.increaseCounter(hash, 1);
        }
    }

    private static String hash() {
        return String.valueOf(Random2.R.nextLong());
    }
}
