#include <dlfcn.h>
#include "jni.h"
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define STRINGIZE2(str) #str
#define STRINGIZE(str) STRINGIZE2(str)

#define WHEREAMI __FILE__ ":" STRINGIZE(__LINE__) ": "

#define local_sprintf(str, format, ...) \
    char str[snprintf(0, 0, format, __VA_ARGS__)]; \
    sprintf(str, format, __VA_ARGS__)

struct library {
    void* handler;
    int (*ctor)(const char* config, void** out);
    void (*dtor)(void* object);
    int (*main)(
        void* object,
        const char* uri,
        const char* metainfo,
        const void* data,
        size_t size,
        void** out);
    void (*logrotate)(void* object);
    void (*free)(void* free);
};

static void throw(JNIEnv* env, const char* class, const char* message) {
    if (!(*env)->ExceptionOccurred(env)) {
        (*env)->ThrowNew(env, (*env)->FindClass(env, class), message);
    }
}

static void cleanup(struct library* library) {
    dlclose(library->handler);
    free(library);
}

static void* load(
    JNIEnv* env,
    struct library* library,
    const char* descr,
    jstring funcName)
{
    const char* name = (*env)->GetStringUTFChars(env, funcName, 0);
    if (!name) {
        cleanup(library);
        local_sprintf(msg, WHEREAMI "Can't get name for %s", descr);
        throw(env, "java/lang/RuntimeException", msg);
        return 0;
    }
    void* ptr = dlsym(library->handler, name);
    (*env)->ReleaseStringUTFChars(env, funcName, name);
    if (!ptr) {
        cleanup(library);
        local_sprintf(
            msg,
            WHEREAMI "Can't load %s from library: %s",
            descr,
            dlerror());
        throw(env, "java/lang/IllegalArgumentException", msg);
        return 0;
    }
    return ptr;
}

static const char* denullify(const char* str) {
    if (str) {
        return str;
    } else {
        return "";
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL Java_ru_yandex_jniwrapper_JniWrapper_load(
    JNIEnv* env,
    jclass class,
    jstring libraryName,
    jstring ctorName,
    jstring dtorName,
    jstring mainName,
    jstring logrotateName,
    jstring freeName)
{
    (void) class;
    struct library* library = calloc(1, sizeof(struct library));
    if (!library) {
        throw(
            env,
            "java/lang/OutOfMemoryError",
            "Can't allocate memory for library");
        return 0;
    }
    const char* name = (*env)->GetStringUTFChars(env, libraryName, 0);
    if (!name) {
        throw(
            env,
            "java/lang/RuntimeException",
            "Can't get library name");
        free(library);
        return 0;
    }
    library->handler = dlopen(name, RTLD_LOCAL | RTLD_NOW);
    (*env)->ReleaseStringUTFChars(env, libraryName, name);
    if (!library->handler) {
        local_sprintf(msg, WHEREAMI "Failed to load library: %s", dlerror());
        throw(env, "java/lang/IllegalArgumentException", msg);
        free(library);
        return 0;
    }
    if (ctorName) {
        library->ctor = load(env, library, "ctor", ctorName);
        if (!library->ctor) {
            return 0;
        }
    }
    if (dtorName) {
        library->dtor = load(env, library, "dtor", dtorName);
        if (!library->dtor) {
            return 0;
        }
    }
    library->main = load(env, library, "main func", mainName);
    if (!library->main) {
        return 0;
    }
    if (logrotateName) {
        library->logrotate = load(env, library, "logrotate", logrotateName);
        if (!library->logrotate) {
            return 0;
        }
    }
    if (freeName) {
        library->free = load(env, library, "free", freeName);
        if (!library->free) {
            return 0;
        }
    } else {
        library->free = free;
    }
    return (jlong) library;
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL Java_ru_yandex_jniwrapper_JniWrapper_unload(
    JNIEnv* env,
    jclass class,
    jlong libraryPtr)
{
    (void) env;
    (void) class;
    cleanup((struct library*) libraryPtr);
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL Java_ru_yandex_jniwrapper_JniWrapper_createInstance(
    JNIEnv* env,
    jclass class,
    jlong libraryPtr,
    jstring configString)
{
    (void) class;
    struct library* library = (struct library*) libraryPtr;
    if (!library->ctor) {
        return 0;
    }
    int rc;
    void* out;
    if (configString) {
        const char* config = (*env)->GetStringUTFChars(env, configString, 0);
        if (!config) {
            throw(
                env,
                "java/lang/RuntimeException",
                "Can't get library name");
            return -1;
        }
        rc = library->ctor(config, &out);
        (*env)->ReleaseStringUTFChars(env, configString, config);
    } else {
        rc = library->ctor(0, &out);
    }
    if (rc == 0) {
        return (jlong) out;
    } else {
        const char* str = denullify(out);
        local_sprintf(
            msg,
            WHEREAMI "Failed to create instance: %s (%p)",
            str,
            out);
        library->free(out);
        throw(env, "java/lang/IllegalArgumentException", msg);
        return 0;
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL Java_ru_yandex_jniwrapper_JniWrapper_destroyInstance(
    JNIEnv* env,
    jclass class,
    jlong libraryPtr,
    jlong objectPtr)
{
    (void) env;
    (void) class;
    struct library* library = (struct library*) libraryPtr;
    if (library->dtor) {
        library->dtor((void*) objectPtr);
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL Java_ru_yandex_jniwrapper_JniWrapper_logrotate(
    JNIEnv* env,
    jclass class,
    jlong libraryPtr,
    jlong objectPtr)
{
    (void) env;
    (void) class;
    struct library* library = (struct library*) libraryPtr;
    if (library->logrotate) {
        library->logrotate((void*) objectPtr);
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jstring JNICALL Java_ru_yandex_jniwrapper_JniWrapper_process(
    JNIEnv* env,
    jclass class,
    jlong libraryPtr,
    jlong objectPtr,
    jstring uriString,
    jstring metainfoString,
    jbyteArray input,
    jsize off,
    jsize len)
{
    (void) class;
    jbyte* buf;
    if (len) {
        buf = malloc(len);
        if (!buf) {
            throw(
                env,
                "java/lang/OutOfMemoryError",
                "Can't allocate memory for buffer");
            return 0;
        }
        (*env)->GetByteArrayRegion(env, input, off, len, buf);
        if ((*env)->ExceptionOccurred(env)) {
            free(buf);
            return 0;
        }
    } else {
        buf = 0;
    }
    const char* uri = (*env)->GetStringUTFChars(env, uriString, 0);
    if (!uri) {
        free(buf);
        throw(env, "java/lang/RuntimeException", "Can't get uri");
        return 0;
    }
    const char* metainfo;
    if (metainfoString) {
        metainfo = (*env)->GetStringUTFChars(env, metainfoString, 0);
        if (!metainfo) {
            (*env)->ReleaseStringUTFChars(env, metainfoString, metainfo);
            free(buf);
            throw(env, "java/lang/RuntimeException", "Can't get metainfo");
            return 0;
        }
    } else {
        metainfo = 0;
    }
    void* out = 0;
    struct library* library = (struct library*) libraryPtr;
    int rc = library->main(
        (void*) objectPtr,
        uri,
        metainfo,
        buf,
        len,
        &out);
    if (metainfo) {
        (*env)->ReleaseStringUTFChars(env, metainfoString, metainfo);
    }
    (*env)->ReleaseStringUTFChars(env, uriString, uri);
    free(buf);
    jstring result = 0;
    if (rc == 0) {
        const char* str = denullify(out);
        result = (*env)->NewStringUTF(env, str);
        if (!result) {
            local_sprintf(
                msg,
                WHEREAMI "Failed to allocate memory for string '%s' (%p)",
                str,
                out);
            throw(env, "java/lang/OutOfMemoryError", msg);
        }
    } else if (rc == -2) {
        throw(env, "java/lang/IllegalArgumentException", out);
    } else {
        throw(env, "java/lang/RuntimeException", out);
    }
    library->free(out);
    return result;
}

