package ru.yandex.chemodan.app.cvdemo.core;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.cvdemo.core.dao.CvSignaturesDao;
import ru.yandex.chemodan.app.cvdemo.core.imageparser.Imageparser;
import ru.yandex.chemodan.app.cvdemo.core.imageparser.TextProcessor;
import ru.yandex.chemodan.app.cvdemo.worker.CalculateImageSignatureHighPriorityTask;
import ru.yandex.chemodan.app.cvdemo.worker.CalculateImageSignatureTask;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class CvSignaturesManager {
    private static final Logger logger = LoggerFactory.getLogger(CvSignaturesManager.class);

    private final CvSignaturesDao signaturesDao;
    private final MpfsClient mpfsClient;
    private final BazingaTaskManager bazingaTaskManager;

    private final Imageparser imageparser;
    private final TextProcessor textProcessor;
    private final MulcaClient mulcaClient;

    public CvSignaturesManager(CvSignaturesDao signaturesDao, MpfsClient mpfsClient,
            BazingaTaskManager bazingaTaskManager, Imageparser imageparser, TextProcessor textProcessor,
            MulcaClient mulcaClient)
    {
        this.signaturesDao = signaturesDao;
        this.mpfsClient = mpfsClient;
        this.bazingaTaskManager = bazingaTaskManager;
        this.imageparser = imageparser;
        this.textProcessor = textProcessor;
        this.mulcaClient = mulcaClient;
    }

    public void reindexUser(String uid) {
        signaturesDao.dropAllForUser(uid);

        File2.withNewTempFile(file -> {
            mpfsClient.diff(uid, "/disk", Option.empty(), response -> {
                //TODO: use new snapshot api
                int statusCode = response.getStatusLine().getStatusCode();
                if (!HttpStatus.is2xx(statusCode)) {
                    // for possible retries
                    throw new HttpException(statusCode, statusCode + " from mpfs");
                }

                try {
                    logger.debug("Storing index.json to {}", file.getAbsolutePath());
                    InputStreamSourceUtils.wrap(response.getEntity().getContent()).readTo(file);
                    logger.debug("Stored index.json to {}", file.getAbsolutePath());
                } catch (IOException e) {
                    throw ExceptionUtils.translate(e);
                }
            });

            try {
                JsonParser parser = (new JsonFactory()).createParser(file.getInputStreamX());

                JsonToken token = parser.nextToken();

                Check.equals(JsonToken.START_OBJECT, token, "root node of mpfs index must be object");

                int rootDepth = 0;
                while (parser.hasCurrentToken()) {
                    if (token.isStructStart()) rootDepth++;
                    if (token.isStructEnd()) rootDepth--;
                    if (rootDepth == 1 && token.equals(JsonToken.FIELD_NAME) && parser.getCurrentName().equals("result")) {
                        parser.nextToken();

                        token = parser.getCurrentToken();

                        Check.equals(JsonToken.START_ARRAY, token, "'result' node must be array");
                        token = parser.nextToken();

                        int depth = 1;
                        while (depth > 0) {
                            if (depth == 1 && token == JsonToken.START_OBJECT) {
                                JsonNode node = new ObjectMapper().readTree(parser);

                                if(node.get("type").asText().equals("file")) {
                                    if ("image".equals(node.get("media_type").asText())) {
                                        String fid = node.get("fid").asText();

                                        //wrong for shared folders
                                        scheduleImageRecalc(uid, uid + ":" + fid, false);
                                    }
                                }
                            } else if (token.isStructStart()) {
                                depth++;
                            } else if (token.isStructEnd()) {
                                depth--;
                            }
                            token = parser.nextToken();
                        }
                    }
                    token = parser.nextToken();
                }
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        });
    }

    public Tuple2List<MpfsFileInfo, Double> findByCv(String uid, String text, int limit) {
        Float[] textSignature = textProcessor.convertText(text);
        ListF<CvSignaturesDao.CvSignatureSearchResult> results = signaturesDao.findNearest(uid, textSignature, limit);
        MapF<String, MpfsFileInfo> byResourceId = mpfsClient
                .bulkInfoByResourceIds(uid, results.map(cv -> cv.resourceId))
                .filter(info -> info.meta.resourceId.isPresent())
                .toMapMappingToKey(info -> info.meta.resourceId.get().toString());

        return results
                .filter(cv -> byResourceId.containsKeyTs(cv.resourceId))
                .toTuple2List(cv -> new Tuple2<>(byResourceId.getTs(cv.resourceId), cv.distance));
    }

    private void scheduleImageRecalc(String uid, String resourceId, boolean online) {
        if (online) {
            bazingaTaskManager.schedule(new CalculateImageSignatureHighPriorityTask(uid, resourceId));
        } else {
            bazingaTaskManager.schedule(new CalculateImageSignatureTask(uid, resourceId));
        }
    }

    public void recalculateSignatureAndStore(String uid, String resourceId) {
        String[] parts = resourceId.split(":");
        mpfsClient
                .getFileInfoOByFileId(uid, parts[0], parts[1])
                .filter(info -> info.path.get().startsWith("/disk/"))
                .filter(info -> info.meta.pmid.isPresent())
                .forEach(info -> {
                    byte[] bytes =
                            mulcaClient.download(MulcaId.fromSerializedString(info.meta.pmid.get())).readBytes();
                    Float[] signature = imageparser.getSignature(bytes);

                    long version = Long.parseLong(info.meta.jsonNode.getField("wh_version").get().getString());

                    signaturesDao.insertOrUpdateSignature(uid, resourceId, version, signature);
                });
    }

    public Float[] textToSignature(String text) {
        return textProcessor.convertText(text);
    }
}
