package ru.yandex.chemodan.app.cvdemo.worker;

import java.io.IOException;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.InputStreamEntity;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.cvdemo.core.CvSignaturesManager;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsStoreOperation;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockFullDescription;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockItemDescription;
import ru.yandex.chemodan.util.TimeUtils;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.exec.ExecUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.Abstract200ResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author tolmalev
 */
public class GenerateGifAnimationTask extends OnetimeTaskSupport<GenerateGifAnimationTask.Parameters> {

    private static final Logger logger = LoggerFactory.getLogger(GenerateGifAnimationTask.class);

    private final CvSignaturesManager manager;
    private final MpfsClient mpfsClient;
    private final MulcaClient mulcaClient;

    public GenerateGifAnimationTask(CvSignaturesManager manager, MpfsClient mpfsClient, MulcaClient mulcaClient) {
        super(Parameters.class);
        this.manager = manager;
        this.mpfsClient = mpfsClient;
        this.mulcaClient = mulcaClient;
    }

    public GenerateGifAnimationTask(String uid, Instant start, Instant end) {
        super(new Parameters(uid, start, end));

        mpfsClient = null;
        manager = null;
        mulcaClient = null;
    }

    @Override
    public void execute(Parameters parameters, ExecutionContext context) throws Exception {
        String resourceId = mpfsClient
                .getFileInfoByUidAndPath(parameters.uid, "/disk/Фотокамера")
                .meta
                .resourceId
                .get()
                .serialize();

        Option<MpfsLentaBlockFullDescription> lentaBlockO =
                mpfsClient.getLentaBlockFilesData(new PassportUid(Long.parseLong(parameters.uid)),
                        resourceId,
                        "image",
                        new MpfsUid(parameters.uid),
                        TimeUtils.unixTime(parameters.start),
                        TimeUtils.unixTimeTill(parameters.end),
                        10000,
                        "");

        if(!lentaBlockO.isPresent()) {
            logger.debug("No files found: {}", parameters);
            return;
        }

        MpfsLentaBlockFullDescription block = lentaBlockO.get();

        ListF<MpfsLentaBlockItemDescription> sortedResources = block.files
                .filter(i -> i.etime.isPresent())
                .filter(i -> i.preview.isPresent())
                .sortedBy(info -> info.etime.get());

        ListF<MpfsLentaBlockItemDescription> data = Cf.arrayList();
        if (sortedResources.isEmpty()) {
            return;
        }

        data.add(sortedResources.get(0));

        for (int i = 1; i < sortedResources.size(); i++) {
            Long prevEtime = sortedResources.get(i - 1).etime.get();
            Long etime = sortedResources.get(i).etime.get();

            if (etime - prevEtime > Duration.standardSeconds(3).getStandardSeconds()) {
                if (data.size() > 3) {
                    logger.debug("Found {} resources in {} - {}", data.size(), parameters.start, parameters.end);
                    generateGif(parameters.uid, data);
                }
                data.clear();
                data.add(sortedResources.get(i));
            } else {
                data.add(sortedResources.get(i));
            }
        }
    }

    private void generateGif(String uid, ListF<MpfsLentaBlockItemDescription> data) {
        ListF<String> resourceIds = data.map(i -> i.resourceId);

        ListF<MpfsFileInfo> fileInfos = mpfsClient.bulkInfoByResourceIds(uid, resourceIds);
        File2.withNewTempDir(dir -> {
            ListF<File2> paths = Cf.arrayList();

            for (MpfsFileInfo info : fileInfos) {
                File2 path = dir.child(Random2.R.nextAlnum(10));
                paths.add(path);

                mulcaClient
                        .download(MulcaId.fromSerializedString(info.meta.pmid.get()))
                        .readTo(path);

                logger.debug("Downloaded file into: {}", path.getAbsolutePath());
            }

            File2 gifPath = dir.child("out.gif");

            ListF<String> command = Cf.arrayList("convert", "-loop", "0", "-delay", "20")
                    .plus(paths.map(File2::getAbsolutePath))
                    .plus(gifPath.getAbsolutePath());

            ExecResult result = ExecUtils.executeGrabbingOutput(command);
            if (result.isSuccess()) {
                String filename = Random2.R.nextAlnum(10) + ".gif";

                MpfsStoreOperation store = mpfsClient
                        .store(uid, "/disk/Загрузки/" + filename, true,
                                Option.empty(), Option.empty(), Option.empty(),
                                Option.empty(), Option.empty(), Option.empty(),
                                false, false, false, null);

                HttpPut httpPut = new HttpPut(store.getUploadUrl().get());
                try {
                    httpPut.setEntity(new InputStreamEntity(gifPath.getInput(), gifPath.length()));
                } catch (IOException e) {
                    throw ExceptionUtils.translate(e);
                }

                ApacheHttpClientUtils.execute(httpPut, new Abstract200ResponseHandler<Void>() {
                    @Override
                    protected Void handle200Response(HttpResponse response)
                    {
                        logger.info("Uploaded gif file {}", filename);
                        return null;
                    }
                }, Timeout.seconds(60));

            } else {
                logger.warn("Failed to generate gif: {}", result.getOutput());
            }
        });
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardHours(1);
    }

    @Override
    public TaskQueueName queueName() {
        return CvDemoTaskQueueName.CV_DEMO_CPU_INTENSIVE;
    }

    @BenderBindAllFields
    public static class Parameters extends DefaultObject {
        private final String uid;
        private final Instant start;
        private final Instant end;

        public Parameters(String uid, Instant start, Instant end) {
            this.uid = uid;
            this.start = start;
            this.end = end;
        }
    }
}
