package ru.yandex.chemodan.app.cvdemo2.admin;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.TimeZone;

import com.fasterxml.jackson.databind.JsonNode;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.mariuszgromada.math.mxparser.Argument;
import org.mariuszgromada.math.mxparser.Expression;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.cvdemo2.core.CvBeautyManager;
import ru.yandex.chemodan.app.cvdemo2.core.dao.CvDataPojo;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.util.auth.YateamAuthUtils;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUser;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistry;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
@ActionContainer
public class CvBeautyAdminPage {
    private static final Logger logger = LoggerFactory.getLogger(CvBeautyAdminPage.class);

    private final CvBeautyManager cvBeautyManager;
    private final YandexStaffUserRegistry staffRegistry;
    private final boolean isPublic;

    public final DynamicProperty<Integer> etimeDelta = new DynamicProperty<Integer>("cvdemo2-etime-delta", 30);
    public final DynamicProperty<Integer> metricLimit = new DynamicProperty<Integer>("cvdemo2-metric-limit", 20);
    public final DynamicProperty<Integer> goodPhotoScore =
            new DynamicProperty<Integer>("cvdemo2-good-metric-limit", 220);
    public final DynamicProperty<Integer> goodPhotoCount =
            new DynamicProperty<Integer>("cvdemo2-good-photos-count", 5);

    public CvBeautyAdminPage(CvBeautyManager cvBeautyManager, YandexStaffUserRegistry staffRegistry, boolean isPublic) {
        this.cvBeautyManager = cvBeautyManager;
        this.staffRegistry = staffRegistry;
        this.isPublic = isPublic;
    }

    @ZAction(defaultAction = true, file = "cv_beauty.xsl")
    @Path("/cv-beauty")
    public CvBeautyResultsPojo cvBeautySearch(
            @RequestParam(value = "uid", required = false) String uid,
            @RequestParam(value = "orderType", required = false) String orderType,
            @RequestParam(value = "metricExpression", required = false) String metricExpression,
            @RequestParam(value = "groupTime", required = false) Option<Integer> groupTime,
            @RequestParam(value = "from", required = false) String from,
            @RequestParam(value = "to", required = false) String to,
            @RequestParam(value = "limit", required = false) Option<Integer> limit,
            @SpecialParam HttpServletRequestX reqX)
    {
        uid = StringUtils.defaultIfEmpty(uid, "").trim();
        String finalOrderType = StringUtils.defaultIfEmpty(orderType, "beautiful").trim();
        from = StringUtils.defaultIfEmpty(from, "").trim();
        to = StringUtils.defaultIfEmpty(to, "").trim();

        Option<String> loginO = YateamAuthUtils.getLoginFromAttributeO(reqX);

        if (loginO.isPresent()) {
            String internalLogin = loginO.getOrThrow(() -> new RuntimeException("Not authorized"));

            YandexStaffUser user = staffRegistry.findByInternalLogin(internalLogin)
                    .getOrThrow(() -> new RuntimeException(internalLogin + ": логин на staff не привязан"));

            uid = Long.toString(user.externalUid.getUid());

            logger.debug("Found ya-team login: {}, external login: {}, uid: {}", internalLogin, user.externalLogin, uid);
        }

        if ("".equals(uid)) {
            return new CvBeautyResultsPojo(uid, orderType, from, to, 20, 300, "", Cf.list(), Cf.list());
        }

        Function<CvDataPojo, Double> rawMetric = d -> {
            if (StringUtils.isEmpty(metricExpression)) {
                return getMetric(finalOrderType, d);
            }
            JsonNode js = JsonNodeUtils.getNode(d.cvJson).get("NeuralNetClasses");

            Argument[] args = Cf.x(js.fieldNames())
                        .map(name -> {
                        return new Argument("_" + name, js.get(name).asDouble());
                    })
                    .toList().toArray(Argument.class);

            Expression expression = new Expression(metricExpression, args);
            return expression.calculate();
        };

        if ("".equals(from) || "".equals(to)) {
            Tuple2<String, String> t2 = detectGoodDateInterval(uid, rawMetric);
            from = t2._1;
            to = t2._2;
        }

        MapF<String, Double> metricByResourceId = cvBeautyManager
                .getCvData(uid, parseDate(from), parseDate(to))
                .toMap(d -> Tuple2.tuple(d.resourceId, rawMetric.apply(d)));

        Function<CvDataPojo, Double> metric = d -> metricByResourceId.getTs(d.resourceId);

        ListF<CvDataPojo> allData = cvBeautyManager
                .getCvData(uid, parseDate(from), parseDate(to))
                .filter(d -> metric.apply(d) > metricLimit.get())
                .sortedBy(d -> d.etime);

        ListF<ListF<CvDataPojo>> groups = Cf.arrayList();

        ListF<CvDataPojo> currentGroup = Cf.arrayList();
        for (int i = 0; i < allData.size(); i++) {
            if (i == 0 || new Duration(allData.get(i-1).etime, allData.get(i).etime)
                    .isLongerThan(Duration.standardSeconds(groupTime.getOrElse(300))))
            {
                if (!currentGroup.isEmpty()) {
                    groups.add(currentGroup);
                }
                currentGroup = Cf.arrayList();
            }
            currentGroup.add(allData.get(i));
        }
        if (!currentGroup.isEmpty()) {
            groups.add(currentGroup);
        }

        ListF<CvDataPojo> selectedData = groups
                .map(g -> g.sortedByDesc(metric).first())
                .sortedByDesc(metric)
                .take(limit.getOrElse(20))
                .sortedBy(d -> d.etime);

        Tuple2List<CvDataPojo, MpfsFileInfo> selected = cvBeautyManager.zipWithFileInfo(selectedData);

        allData = allData.sortedByDesc(metric).take(200);
        Tuple2List<CvDataPojo, MpfsFileInfo> data = cvBeautyManager.zipWithFileInfo(allData);

        Function<Tuple2<CvDataPojo, MpfsFileInfo>, CvResultPojo> mapper =
                t -> {
                    MapF<String, String> previews =
                            t._2.getMeta().getMetaJsonField("sizes").get().getArrayElements().toMap(js -> {
                                return new Tuple2<>(js.getField("name").get().getValueAsString(),
                                        js.getField("url").get().getValueAsString());
                            });

                    return new CvResultPojo(previews.getTs("L"), previews.getTs("XXXL"),
                            metric.apply(t._1), t._1.cvJson);
                };

        return new CvBeautyResultsPojo(uid, finalOrderType, from, to, limit.getOrElse(20),
                groupTime.getOrElse(300), metricExpression, selected.map(mapper), data.map(mapper));
    }

    private Tuple2<String, String> detectGoodDateInterval(String uid, Function<CvDataPojo, Double> rawMetric) {
        Instant now = Instant.now();

        Instant foundEnd = now;
        Instant foundStart = now.minus(Duration.standardDays(30));
        boolean found = false;

        int daysInInterval = 21;

        for (int i = 0; i * daysInInterval < 365; i++) {
            Instant end = now.minus(Duration.standardDays(i * daysInInterval));
            Instant start = now.minus(Duration.standardDays((i + 1) * daysInInterval));

            ListF<Double> metrics = cvBeautyManager
                    .getCvData(uid, start, end, Option.of(500))
                    .map(rawMetric);

            if (!found && metrics.size() > 5) {
                foundEnd = end;
                foundStart = start;
            }

            ListF<Double> goodMetrics = metrics
                    .filter(s -> s > goodPhotoScore.get());

            if (goodMetrics.size() > goodPhotoCount.get()) {
                foundEnd = end;
                foundStart = start;
                break;
            }
        }
        return new Tuple2<>(formatDate(foundStart), formatDate(foundEnd));
    }

    private double getMetric(String finalOrderType, CvDataPojo d) {
        try {
            return JsonNodeUtils.getNode(d.cvJson).get("NeuralNetClasses").get(finalOrderType).asDouble();
        } catch (RuntimeException e) {
            try {
                return JsonNodeUtils.getNode(d.cvJson).get(finalOrderType).asDouble();
            } catch (RuntimeException e2) {
                return -1000;
            }
        }
    }

    public static String formatDate(Instant date) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        simpleDateFormat.setTimeZone(TimeZone.getTimeZone("Europe/Moscow"));

        return simpleDateFormat.format(date.toDate());
    }

    Instant parseDate(String dateStr) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        simpleDateFormat.setTimeZone(TimeZone.getTimeZone("Europe/Moscow"));

        try {
            return new Instant(simpleDateFormat.parse(dateStr)).plus(Duration.standardHours(2));
        } catch (ParseException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Bendable
    @XmlRootElement(name = "content")
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    private static class CvBeautyResultsPojo {
        public final String uid;
        public final String orderType;
        public final int groupTime;
        public final String metricExpression;
        public final String from;
        public final String to;
        public final int limit;

        public final ListF<CvResultsRow> selected;
        public final ListF<CvResultsRow> all;

        public final String appJson;

        private CvBeautyResultsPojo(String uid, String orderType, String from, String to, int limit,
                int groupTime, String metricExpression, ListF<CvResultPojo> selected, ListF<CvResultPojo> all)
        {
            this.uid = uid;
            this.orderType = orderType;
            this.from = from;
            this.to = to;
            this.limit = limit;
            this.groupTime = groupTime;
            this.metricExpression = metricExpression;
            this.selected = selected.paginate(5).map(CvResultsRow::new);
            this.all = all.paginate(5).map(CvResultsRow::new);

            appJson = "["
                    + selected.map(s -> "{"
                        + "\"preview-url\": \"" + UrlUtils.removeParameter(s.bigPreviewUrl, "size") + "\""
                        + ",\"score\": \"" + s.metricValue + "\""
                    + "}").mkString(",")
                    + "]";
        }
    }

    @BenderBindAllFields
    private static final class CvResultsRow {
        public final ListF<CvResultPojo> items;

        private CvResultsRow(ListF<CvResultPojo> items) {
            this.items = items;
        }
    }

    @BenderBindAllFields
    private static class CvResultPojo {
        public final String previewUrl;
        public final String bigPreviewUrl;
        public final double metricValue;
        public final String allJsonData;

        private CvResultPojo(String previewUrl, String bigPreviewUrl, double metricValue, String allJsonData) {
            this.previewUrl = previewUrl;
            this.bigPreviewUrl = bigPreviewUrl;
            this.metricValue = metricValue;
            this.allJsonData = allJsonData;
        }
    }
}
