package ru.yandex.chemodan.app.cvdemo2.admin;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import lombok.Data;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.util.auth.YateamAuthUtils;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUser;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistry;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author yashunsky
 */
@AllArgsConstructor
@ActionContainer
public class CvDocumentsAdminPage {

    private static final int DEFAULT_LIMIT = 100;

    private final static String TEMPLATE_FILE = "documents_request.txt";

    private static final ObjectMapper mapper = new ObjectMapper();

    private final YandexStaffUserRegistry staffRegistry;
    private final HttpClient diskSearchHttpClient;
    private final String searchHost;
    private final MpfsClient mpfsClient;

    @ZAction(defaultAction = true, file = "cv_documents.xsl")
    @Path("/cv-documents")
    public DocumentsResultPojo findDocuments(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam(value = "limit", required = false) Option<Integer> limit)
    {
        String internalLogin = YateamAuthUtils.getLoginFromAttributeO(reqX).getOrThrow(() -> new RuntimeException("Not authorized"));
        YandexStaffUser user = staffRegistry.findByInternalLogin(internalLogin)
                .getOrThrow(() -> new RuntimeException(internalLogin + ": логин на staff не привязан"));

        String uid = Long.toString(user.externalUid.getUid());

        String url = generateUrl(uid);

        ListF<String> previewStids = ApacheHttpClientUtils.execute(new HttpGet(url), diskSearchHttpClient, response -> {
            JsonNode node = mapper.readTree(response.getEntity().getContent());
            node = node.get("hitsArray");
            Check.notNull(node, "bad response from search: no 'hitsArray'");
            Check.isTrue(node.isArray(), "hitsArray field must be array");
            ListF<String> result = Cf.arrayList();

            for (int i = 0; i < node.size(); i++) {
                JsonNode partNode = node.get(i);
                Check.notNull(partNode.get("preview_stid"), "no field 'word' in response");
                result.add(partNode.get("preview_stid").asText());
            }
            return result;
        });

        ListF<String> links = previewStids.take(limit.getOrElse(DEFAULT_LIMIT)).map(stid -> mpfsClient.generateZaberunUrl(
                stid, stid, "preview",
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.of("400x400"),
                Option.empty(),
                Option.empty(),
                Option.empty(),
                Option.of(1),
                Option.of(true),
                Option.empty()));

        return new DocumentsResultPojo(links.map(DocInfo::new));
    }

    private String generateUrl(String uid) {
        String template = new ClassPathResourceInputStreamSource(CvDocumentsAdminPage.class, TEMPLATE_FILE)
                .readLines().firstO().getOrThrow("request template not found");

        return searchHost + template.replaceFirst("%PREFIX%", uid);
    }

    @Data
    @BenderBindAllFields
    @XmlRootElement(name = "content")
    static class DocumentsResultPojo {
        public final ListF<DocInfo> docs;
    }

    @Data
    @BenderBindAllFields
    static class DocInfo {
        public final String previewLink;
    }
}
