package ru.yandex.chemodan.app.cvdemo2.admin;

import java.util.List;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchClient;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.auth.YateamAuthUtils;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUser;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistry;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
@ActionContainer
public class CvGridAdminPage2 {
    private static final Logger logger = LoggerFactory.getLogger(CvGridAdminPage2.class);

    private final DiskSearchClient diskSearchClient;
    private final MpfsClient mpfsClient;
    private final YandexStaffUserRegistry staffRegistry;

    private final DynamicProperty<Integer> normalPhotoSize = new DynamicProperty<>("cv-grid-normalPhotoSize", 8);

    private final DynamicProperty<Double> beautyK = new DynamicProperty<>("cv-grid-beautyK", 0.25);
    private final DynamicProperty<Double> normalSizeK = new DynamicProperty<>("cv-grid-normalSizeK", 0.25);
    private final DynamicProperty<Double> cropK = new DynamicProperty<>("cv-grid-cropK", 0.0);
    private final DynamicProperty<Double> sizeDiffK = new DynamicProperty<>("cv-grid-sizeDiffK", 0.25);
    private final DynamicProperty<Double> photosInBlockK = new DynamicProperty<>("cv-grid-photosInBlockK", 0.25);
    private final DynamicProperty<Double> singlePhotoK = new DynamicProperty<>("cv-grid-singlePhotoK", -100.0);

    private final MapF<String, MapF<String, List<GridGenerator.BaseGrid>>> generatedBaseGrids = Cf.hashMap();

    public CvGridAdminPage2(DiskSearchClient diskSearchClient, MpfsClient mpfsClient, YandexStaffUserRegistry staffRegistry) {
        this.diskSearchClient = diskSearchClient;
        this.mpfsClient = mpfsClient;
        this.staffRegistry = staffRegistry;

        MapF<String, int[][]> mobileAllowedByType = Cf.hashMap();
        mobileAllowedByType.put("h", new int[][] {
                {2, 2},
                {3, 3},
                {4, 4},
                {6, 6}
        });

        mobileAllowedByType.put("v", new int[][] {
                {2, 4},
                {3, 5},
                {4, 6},
                {6, 10}
        });

        mobileAllowedByType.put("s", new int[][] {
                {3, 4},
                {6, 8},
        });

        mobileAllowedByType.put("-", new int[][] {
                {4, 2},
                {6, 3}
        });

        MapF<String, int[][]> webAllowedByType = Cf.hashMap();
        webAllowedByType.put("h", new int[][] {
                {2, 2},
                {3, 3},
                {4, 4},
                {5, 5},
                {6, 6}
        });

        webAllowedByType.put("s", new int[][] {
                {2, 3},
                {4, 6},
        });

        webAllowedByType.put("v", new int[][] {
                {2, 4},
                {3, 5},
                {4, 8},
        });

        webAllowedByType.put("-", new int[][] {
                {4,2},
                {6,3},
                {8,4},
        });

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(6, 6, mobileAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(6, 6, mobileAllowedByType));

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(10, 7, webAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(10, 7, webAllowedByType));

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(10, 8, webAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(10, 8, webAllowedByType));

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(10, 9, webAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(10, 9, webAllowedByType));

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(12, 7, webAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(12, 7, webAllowedByType));

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(12, 8, webAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(12, 8, webAllowedByType));

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(12, 9, webAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(12, 9, webAllowedByType));

        mobileAllowedByType.put("h", new int[][] {
                {2, 2},
                {3, 3},
                {4, 4},
                {5, 5},
                {6, 6},
                {7, 7},
        });

        mobileAllowedByType.put("s", new int[][] {
                {2, 3},
                {4, 6},
                {6, 9},
        });

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(14, 7, webAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(14, 7, webAllowedByType));

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(14, 8, webAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(14, 8, webAllowedByType));

        generatedBaseGrids.put(
                new BaseGridGenerator.BaseGridGeneratorConfig(14, 9, webAllowedByType).toString(),
                new BaseGridGenerator().generateBaseGrids(14, 9, webAllowedByType));
    }

    @Data
    static class PhotoPosition {
        public final int x;
        public final int y;
        public final int w;
        public final int h;
    }

    @ZAction(defaultAction = true, file = "cv_grid.xsl")
    @Path("/cv-grid2")
    public CvGridsPojo cvBeautySearch(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam(value = "count", required = false) Option<Integer> count,
            @RequestParam(value = "daysAgoStart", required = false) Option<Integer> daysAgoStart,
            @RequestParam(value = "daysAgoEnd", required = false) Option<Integer> daysAgoEnd,
            @RequestParam(value = "uid", required = false) Option<String> suid,
            @RequestParam(value = "cellW", required = false) Option<Integer> cellW,

            @RequestParam(value = "normalPhotoSize", required = false) Option<Integer> normalPhotoSize,
            @RequestParam(value = "normalSizeK", required = false) Option<Double> normalSizeK,
            @RequestParam(value = "beautyK", required = false) Option<Double> beautyK,
            @RequestParam(value = "cropK", required = false) Option<Double> cropK,
            @RequestParam(value = "sizeDiffK", required = false) Option<Double> sizeDiffK,
            @RequestParam(value = "photosInBlockK", required = false) Option<Double> photosInBlockK,
            @RequestParam(value = "singlePhotoK", required = false) Option<Double> singlePhotoK,

            @RequestParam(value = "onlyVertical", required = false) Option<Boolean> onlyVertical,
            @RequestParam(value = "onlyHorizontal", required = false) Option<Boolean> onlyHorizontal,

            @RequestParam(value = "gridColumns", required = false) Option<Integer> gridColumns,
            @RequestParam(value = "maxPhotosInBlock", required = false) Option<Integer> maxPhotosInBlock,

            @RequestParam(value = "verticalSizes", required = false) Option<String> verticalSizes,
            @RequestParam(value = "horizontalSizes", required = false) Option<String> horizontalSizes,
            @RequestParam(value = "panoramaSizes", required = false) Option<String> panoramaSizes,
            @RequestParam(value = "squareSizes", required = false) Option<String> squareSizes
    ) {
        PassportUid finalUid = PassportUid.cons(Cf.Long.parse(suid.getOrElse("50273844")));
        Option<String> loginO = YateamAuthUtils.getLoginFromAttributeO(reqX);

        if (loginO.isPresent()) {
            String internalLogin = loginO.getOrThrow(() -> new RuntimeException("Not authorized"));

            YandexStaffUser user = staffRegistry.findByInternalLogin(internalLogin)
                    .getOrThrow(() -> new RuntimeException(internalLogin + ": логин на staff не привязан"));

            finalUid = user.externalUid;

            logger.debug("Found ya-team login: {}, external login: {}, uid: {}", internalLogin, user.externalLogin, finalUid);
        }

        PassportUid uid = finalUid;

        Integer finalCount = count.getOrElse(100);
        Integer finalDaysAgoStart = daysAgoStart.getOrElse(365);
        Integer finalDaysAgoEnd = daysAgoEnd.getOrElse(0);

        int finalMaxPhotosInBlock = maxPhotosInBlock.getOrElse(7);

        Integer finalNormalPhotoSize = normalPhotoSize.getOrElse(this.normalPhotoSize.get());
        Double finalBeautyK = beautyK.getOrElse(this.beautyK.get());
        Double finalNormalSizeK = normalSizeK.getOrElse(this.normalSizeK.get());
        Double finalCropK = cropK.getOrElse(this.cropK.get());
        Double finalSizeDiffK = sizeDiffK.getOrElse(this.sizeDiffK.get());
        Double finalPhotosInBlockK = photosInBlockK.getOrElse(this.photosInBlockK.get());
        Double finalSinglePhotoK = singlePhotoK.getOrElse(this.singlePhotoK.get());

        String finalHorizontalSizes = horizontalSizes.getOrElse("2,2|3,3|4,4|5,5|6,6");
        String finalVerticalSizes = verticalSizes.getOrElse("2,4|3,5|4,8");
        String finalSquareSizes = squareSizes.getOrElse("2,3|4,6");
        String finalPanoramaSizes = panoramaSizes.getOrElse("4,2|6,3|8,4");

        MapF<String, int[][]> allowedSizesByType = Cf.hashMap();

        allowedSizesByType.put("v", Cf.list(finalVerticalSizes.split("\\|"))
                .map(s -> Cf.x(s.trim().split(",")).map(Cf.Integer::parse).mapToIntArray(i -> i)).toArray(new int[][]{}));
        allowedSizesByType.put("h", Cf.list(finalHorizontalSizes.split("\\|"))
                .map(s -> Cf.x(s.trim().split(",")).map(Cf.Integer::parse).mapToIntArray(i -> i)).toArray(new int[][]{}));
        allowedSizesByType.put("s", Cf.list(finalSquareSizes.split("\\|"))
                .map(s -> Cf.x(s.trim().split(",")).map(Cf.Integer::parse).mapToIntArray(i -> i)).toArray(new int[][]{}));
        allowedSizesByType.put("-", Cf.list(finalPanoramaSizes.split("\\|"))
                .map(s -> Cf.x(s.trim().split(",")).map(Cf.Integer::parse).mapToIntArray(i -> i)).toArray(new int[][]{}));

        BaseGridGenerator.BaseGridGeneratorConfig baseGridGeneratorConfig = new BaseGridGenerator.BaseGridGeneratorConfig(
                gridColumns.getOrElse(6),
                finalMaxPhotosInBlock,
                allowedSizesByType
        );

        MapF<String, List<GridGenerator.BaseGrid>> baseGrids =
                generatedBaseGrids.getOrElseUpdate(baseGridGeneratorConfig.toString(), () -> new BaseGridGenerator().generateBaseGrids(baseGridGeneratorConfig));

        GridGenerator gridGenerator = new GridGenerator(new GridGenerator.GridGeneratorConfig(
                baseGrids,
                finalNormalPhotoSize,
                finalBeautyK,
                finalNormalSizeK,
                finalCropK,
                finalSizeDiffK,
                finalPhotosInBlockK,
                finalSinglePhotoK,
                Option.of('s')
        ));

        InstantInterval interval = new InstantInterval(
                Instant.now().minus(Duration.standardDays(finalDaysAgoStart)),
                Instant.now().minus(Duration.standardDays(finalDaysAgoEnd))
        );

        ListF<DiskSearchFileInfo> searchInfos = diskSearchClient
                .findPhotosWithBeauty(uid, interval, Option.empty(), 0, 10000).hitsArray
                        .filter(sfi -> sfi.width.isPresent()
                                && sfi.height.isPresent()
                                && sfi.getBeautiful2().isPresent()
                                && (
                                        !onlyVertical.getOrElse(false) || sfi.height.get() > sfi.width.get()
                                )
                                && (
                                        !onlyHorizontal.getOrElse(false) || sfi.height.get() < sfi.width.get()
                                )
                        ).sortedByDesc(f -> f.etime.get())
                        .take(finalCount * 3);

        MapF<String, DiskSearchFileInfo> fiByResourceId = searchInfos.toMapMappingToKey(item -> uid.toString() + ":" + item.getId());

        ListF<String> resourceIds = searchInfos.map(item -> uid.toString() + ":" + item.getId());
        ListF<MpfsFileInfo> files = mpfsClient.bulkInfoByResourceIds(MpfsUser.of(uid), resourceIds,
                Cf.list("/disk", "/photounlim"))
                .sortedByDesc(f -> fiByResourceId.getTs(f.getMeta().getResourceId().get().toString()).etime.get())
                .take(finalCount);


        ListF<GridGenerator.PhotoInfo> photoInfos = files.map(f -> {
            DiskSearchFileInfo fi = fiByResourceId.getTs(f.getMeta().getResourceId().get().toString());

            return new GridGenerator.PhotoInfo(
                    fi.width.get(),
                    fi.height.get(),
                    fi.getBeautiful2()
            );
        });

        Instant start = Instant.now();
        GridGenerator.Grid mobileGrid = gridGenerator.generateGrid(photoInfos);
        System.out.println("mobile grid generation time : " + TimeUtils.secondsStringToNow(start) + " photos count " + files.size());

        int minX = 500;
        int minY = 100;
        int gridCellW = cellW.getOrElse(60);
        int gridCellH = gridCellW * 2 / 3;

        ListF<OneImageGridPojo> mobileImages = Cf.arrayList();
        for (int i = 0; i < files.size(); i++) {
            MpfsFileInfo info = files.get(i);
            GridGenerator.PhotoPosition pos = mobileGrid.positions.get(i);

            mobileImages.add(new OneImageGridPojo(
                    minX + gridCellW * pos.x,
                    minY + gridCellH * pos.y,
                    pos.w * gridCellW,
                    pos.h * gridCellH,
                    info.getMeta().getPreview().get(),
                    info.getMeta().getFileUrl().get()
            ));
        }

        return new CvGridsPojo(
                uid.toString(),
                finalCount,

                finalDaysAgoStart,
                finalDaysAgoEnd,

                finalNormalPhotoSize,
                finalNormalSizeK,
                finalBeautyK,
                finalCropK,
                finalSizeDiffK,
                finalPhotosInBlockK,
                finalSinglePhotoK,

                onlyVertical.getOrElse(false),
                onlyHorizontal.getOrElse(false),

                cellW.getOrElse(60),
                gridColumns.getOrElse(6),
                finalMaxPhotosInBlock,

                finalVerticalSizes,
                finalHorizontalSizes,
                finalPanoramaSizes,
                finalSquareSizes,

                Cf.list(new CvGridPojo(mobileImages))
        );
    }

    @Data
    @BenderBindAllFields
    @XmlRootElement(name = "content")
    static class CvGridsPojo {
        public final String uid;
        public final Integer count;

        public final Integer daysAgoStart;
        public final Integer daysAgoEnd;

        public final Integer normalPhotoSize;
        public final Double normalSizeK;
        public final Double beautyK;
        public final Double cropK;
        public final Double sizeDiffK;
        public final Double photosInBlockK;
        public final Double singlePhotoK;

        public final boolean onlyVertical;
        public final boolean onlyHorizontal;

        public final int cellWidth;
        public final int columns;
        public final int maxPhotosInBlock;

        public final String verticalSizes;
        public final String horizontalSizes;
        public final String panoramaSizes;
        public final String squareSizes;

        public final ListF<CvGridPojo> grids;
    }

    @Data
    @BenderBindAllFields
    @XmlRootElement(name = "content")
    static class CvGridPojo {
        public final ListF<OneImageGridPojo> images;
    }

    @Data
    @BenderBindAllFields
    static class OneImageGridPojo {
        public final int x;
        public final int y;
        public final int w;
        public final int h;
        public final String previewLink;
        public final String originalLink;
    }
}
